#!/usr/bin/env pwsh

Write-Host "Verifying kernel32.dll exports..." -ForegroundColor Cyan
Write-Host ""

$apis = @(
    'GetComputerNameW',
    'GetLocalTime',
    'GetDiskFreeSpaceW',
    'GetModuleFileNameW',
    'LoadLibraryW',
    'GetProcAddress',
    'VirtualAlloc',
    'VirtualFree',
    'CreateThread',
    'GetTickCount',
    'Sleep',
    'GetCurrentProcessId',
    'GetCurrentThreadId',
    'GetSystemDirectoryW',
    'GetWindowsDirectoryW'
)

Add-Type -TypeDefinition @"
using System;
using System.Runtime.InteropServices;

public class Kernel32Checker {
    [DllImport("kernel32.dll", CharSet = CharSet.Unicode)]
    public static extern IntPtr LoadLibrary(string lpFileName);

    [DllImport("kernel32.dll", CharSet = CharSet.Ansi)]
    public static extern IntPtr GetProcAddress(IntPtr hModule, string lpProcName);
}
"@

$kernel32Handle = [Kernel32Checker]::LoadLibrary("kernel32.dll")

if ($kernel32Handle -eq [IntPtr]::Zero) {
    Write-Host "ERROR: Could not load kernel32.dll" -ForegroundColor Red
    exit 1
}

$found = 0
$notFound = 0

foreach ($api in $apis) {
    $addr = [Kernel32Checker]::GetProcAddress($kernel32Handle, $api)
    if ($addr -ne [IntPtr]::Zero) {
        Write-Host "[OK] $api" -ForegroundColor Green
        $found++
    } else {
        Write-Host "[FAIL] $api NOT FOUND" -ForegroundColor Red
        $notFound++
    }
}

Write-Host ""
Write-Host "Summary:" -ForegroundColor Cyan
Write-Host "  Found: $found" -ForegroundColor Green
Write-Host "  Not Found: $notFound" -ForegroundColor $(if ($notFound -gt 0) { "Red" } else { "Green" })

if ($notFound -gt 0) {
    exit 1
}
