#!/usr/bin/env pwsh
# Test all recipes with the specified output directory

$cli = "X:\Projects\dotnet\wildfire-buffet\WildfireBuffet.CLI\bin\Release\net8.0\WildfireBuffet.CLI.exe"
$outputDir = "X:\Projects\dotnet\wildfire-buffet\WildfireBuffet.UI\bin\Release\net8.0\win-x64\output"
$recipesDir = "X:\Projects\dotnet\wildfire-buffet\Recipes"

Write-Host "`n=== Testing All Recipes with Output Directory ===" -ForegroundColor Cyan
Write-Host "Output Directory: $outputDir`n" -ForegroundColor Yellow

# Get all recipe files
$recipes = Get-ChildItem "$recipesDir\*.json" | Sort-Object Name

$results = @()

foreach ($recipe in $recipes) {
    Write-Host "`n[$($recipe.Name)]" -ForegroundColor Green -NoNewline
    Write-Host " Building..." -NoNewline

    $output = & $cli --config $recipe.FullName --output $outputDir 2>&1

    if ($output -match "1 succeeded" -or $output -match "Success") {
        Write-Host " [SUCCESS]" -ForegroundColor Green
        $results += @{ Recipe = $recipe.Name; Status = "SUCCESS" }
    }
    elseif ($output -match "Failed|Error") {
        Write-Host " [FAILED]" -ForegroundColor Red
        $error = ($output | Select-String -Pattern "Error:|Failed:").Line | Select-Object -First 1
        Write-Host "  Error: $error" -ForegroundColor DarkRed
        $results += @{ Recipe = $recipe.Name; Status = "FAILED"; Error = $error }
    }
    else {
        Write-Host " ? UNKNOWN" -ForegroundColor Yellow
        $results += @{ Recipe = $recipe.Name; Status = "UNKNOWN" }
    }
}

Write-Host "`n`n=== Summary ===" -ForegroundColor Cyan
$successCount = ($results | Where-Object { $_.Status -eq "SUCCESS" }).Count
$failedCount = ($results | Where-Object { $_.Status -eq "FAILED" }).Count
$unknownCount = ($results | Where-Object { $_.Status -eq "UNKNOWN" }).Count

Write-Host "Total Recipes: $($recipes.Count)" -ForegroundColor White
Write-Host "[+] Succeeded: $successCount" -ForegroundColor Green
Write-Host "[-] Failed: $failedCount" -ForegroundColor Red
Write-Host "[?] Unknown: $unknownCount" -ForegroundColor Yellow

# Show output directory contents
Write-Host "`n`n=== Output Directory Contents ===" -ForegroundColor Cyan
if (Test-Path $outputDir) {
    $files = Get-ChildItem $outputDir | Sort-Object LastWriteTime -Descending | Select-Object -First 20
    foreach ($file in $files) {
        $size = if ($file.Length -gt 1MB) { "{0:N2} MB" -f ($file.Length / 1MB) }
                elseif ($file.Length -gt 1KB) { "{0:N2} KB" -f ($file.Length / 1KB) }
                else { "$($file.Length) B" }
        Write-Host "  $($file.Name) - $size" -ForegroundColor Gray
    }
}
else {
    Write-Host "  Output directory not found!" -ForegroundColor Red
}
