#!/usr/bin/env pwsh
<#
.SYNOPSIS
    Runs tests with code coverage collection and generates HTML reports

.DESCRIPTION
    This script runs all tests in the solution with Coverlet code coverage,
    then generates detailed HTML reports using ReportGenerator.

.PARAMETER Configuration
    Build configuration (Debug or Release). Default: Debug

.PARAMETER OpenReport
    Opens the coverage report in the default browser after generation

.PARAMETER Threshold
    Minimum coverage percentage threshold (0-100). Build fails if below threshold.

.PARAMETER Format
    Coverage output formats (comma-separated): cobertura,opencover,lcov,json
    Default: cobertura,opencover

.EXAMPLE
    .\Scripts\Run-CodeCoverage.ps1
    Runs coverage with default settings

.EXAMPLE
    .\Scripts\Run-CodeCoverage.ps1 -Configuration Release -OpenReport
    Runs coverage in Release mode and opens the report

.EXAMPLE
    .\Scripts\Run-CodeCoverage.ps1 -Threshold 80
    Runs coverage and fails if coverage is below 80%
#>

param(
    [ValidateSet('Debug', 'Release')]
    [string]$Configuration = 'Debug',

    [switch]$OpenReport,

    [ValidateRange(0, 100)]
    [int]$Threshold = 0,

    [string]$Format = 'cobertura,opencover'
)

# Script configuration
$ErrorActionPreference = 'Stop'
$ScriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$RootDir = Split-Path -Parent $ScriptDir
$TestProject = Join-Path $RootDir "WildfireBuffet.Tests\WildfireBuffet.Tests.csproj"
$CoverageDir = Join-Path $RootDir "coverage"
$CoverageResultsDir = Join-Path $CoverageDir "results"
$CoverageReportDir = Join-Path $CoverageDir "report"

Write-Host "🧪 Wildfire Buffet - Code Coverage Runner" -ForegroundColor Cyan
Write-Host "=========================================" -ForegroundColor Cyan
Write-Host ""

# Clean previous coverage results
if (Test-Path $CoverageDir) {
    Write-Host "Cleaning previous coverage results..." -ForegroundColor Yellow
    Remove-Item -Path $CoverageDir -Recurse -Force
}

# Create coverage directories
New-Item -ItemType Directory -Path $CoverageResultsDir -Force | Out-Null
New-Item -ItemType Directory -Path $CoverageReportDir -Force | Out-Null

# Build the solution
Write-Host "Building solution..." -ForegroundColor Yellow
dotnet build "$RootDir\WildfireBuffet.sln" -c $Configuration --nologo
if ($LASTEXITCODE -ne 0) {
    Write-Host "❌ Build failed" -ForegroundColor Red
    exit $LASTEXITCODE
}
Write-Host "✅ Build successful" -ForegroundColor Green
Write-Host ""

# Run tests with coverage
Write-Host "Running tests with coverage collection..." -ForegroundColor Yellow
Write-Host "  Configuration: $Configuration" -ForegroundColor Gray
Write-Host "  Output format: $Format" -ForegroundColor Gray
Write-Host ""

$coverageFile = Join-Path $CoverageResultsDir "coverage.xml"

# Coverlet settings
$coverletArgs = @(
    "test",
    $TestProject,
    "-c", $Configuration,
    "--no-build",
    "--nologo",
    "/p:CollectCoverage=true",
    "/p:CoverletOutputFormat=$Format",
    "/p:CoverletOutput=`"$CoverageResultsDir/`"",
    "/p:ExcludeByFile=`"**/*Designer.cs,**/*.g.cs,**/*.g.i.cs`"",
    "/p:ExcludeByAttribute=`"Obsolete,GeneratedCode,CompilerGenerated`"",
    "/p:Exclude=`"[*.Tests]*,[*.UI.Views]*,[xunit*]*`""
)

# Add threshold if specified
if ($Threshold -gt 0) {
    $coverletArgs += "/p:Threshold=$Threshold"
    $coverletArgs += "/p:ThresholdType=line"
    Write-Host "  Threshold: $Threshold%" -ForegroundColor Gray
}

# Run tests
& dotnet @coverletArgs

$testExitCode = $LASTEXITCODE

if ($testExitCode -ne 0) {
    Write-Host ""
    Write-Host "❌ Tests failed or coverage threshold not met" -ForegroundColor Red
    exit $testExitCode
}

Write-Host ""
Write-Host "✅ Tests passed" -ForegroundColor Green
Write-Host ""

# Generate HTML report
Write-Host "Generating coverage report..." -ForegroundColor Yellow

# Find the coverage file (Coverlet may create it with different extensions)
$coverageFiles = Get-ChildItem -Path $CoverageResultsDir -Filter "coverage.*" -File

if ($coverageFiles.Count -eq 0) {
    Write-Host "⚠️  No coverage files found in $CoverageResultsDir" -ForegroundColor Yellow
    exit 1
}

Write-Host "  Found $($coverageFiles.Count) coverage file(s)" -ForegroundColor Gray

# Generate report using ReportGenerator
$reportGenArgs = @(
    "-reports:$CoverageResultsDir\coverage.*",
    "-targetdir:$CoverageReportDir",
    "-reporttypes:Html;HtmlSummary;Badges;TextSummary",
    "-classfilters:-System.*;-Microsoft.*;-Avalonia.*;-xunit.*"
)

& dotnet reportgenerator @reportGenArgs

if ($LASTEXITCODE -ne 0) {
    Write-Host "❌ Report generation failed" -ForegroundColor Red
    exit $LASTEXITCODE
}

Write-Host "✅ Report generated" -ForegroundColor Green
Write-Host ""

# Display summary
$summaryFile = Join-Path $CoverageReportDir "Summary.txt"
if (Test-Path $summaryFile) {
    Write-Host "📊 Coverage Summary:" -ForegroundColor Cyan
    Write-Host "===================" -ForegroundColor Cyan
    Get-Content $summaryFile | Write-Host
    Write-Host ""
}

# Show report location
$reportIndex = Join-Path $CoverageReportDir "index.html"
Write-Host "📁 Coverage report saved to:" -ForegroundColor Green
Write-Host "   $reportIndex" -ForegroundColor White
Write-Host ""

# Badge files
$badgeDir = $CoverageReportDir
$lineCoverageBadge = Join-Path $badgeDir "badge_linecoverage.svg"
$branchCoverageBadge = Join-Path $badgeDir "badge_branchcoverage.svg"

if (Test-Path $lineCoverageBadge) {
    Write-Host "🏆 Coverage badges:" -ForegroundColor Green
    Write-Host "   Line: $lineCoverageBadge" -ForegroundColor White
    Write-Host "   Branch: $branchCoverageBadge" -ForegroundColor White
    Write-Host ""
}

# Open report if requested
if ($OpenReport) {
    Write-Host "Opening coverage report in browser..." -ForegroundColor Yellow
    Start-Process $reportIndex
}

Write-Host "✅ Coverage analysis complete!" -ForegroundColor Green

# Return exit code based on threshold
exit 0
