# Package Wildfire Buffet CLI with Native AOT for distribution
param(
    [string]$OutputPath = ".\dist"
)

$ErrorActionPreference = "Stop"

Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Wildfire Buffet CLI Packaging" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host ""

# Check for C++ build tools
Write-Host "[1/6] Checking C++ build tools..." -ForegroundColor Yellow
$vswhere = "${env:ProgramFiles(x86)}\Microsoft Visual Studio\Installer\vswhere.exe"
$hasCppTools = $false

if (Test-Path $vswhere) {
    $vsPath = & $vswhere -latest -requires Microsoft.VisualStudio.Component.VC.Tools.x86.x64 -property installationPath
    if ($vsPath) {
        $hasCppTools = $true
        Write-Host "  C++ build tools found: $vsPath" -ForegroundColor Green
    }
}

if (-not $hasCppTools) {
    Write-Host ""
    Write-Host "ERROR: C++ Build Tools Not Found" -ForegroundColor Red
    Write-Host ""
    Write-Host "Native AOT compilation REQUIRES Visual Studio C++ build tools." -ForegroundColor Yellow
    Write-Host ""
    Write-Host "Download and install:" -ForegroundColor White
    Write-Host "  https://visualstudio.microsoft.com/downloads/#build-tools-for-visual-studio-2022" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "Then select 'Desktop development with C++' workload" -ForegroundColor Gray
    Write-Host ""
    exit 1
}

# Setup paths
$ProjectRoot = Split-Path -Parent (Split-Path -Parent $PSCommandPath)
$OutputPath = Join-Path $ProjectRoot $OutputPath
if (Test-Path $OutputPath) {
    Remove-Item -Path $OutputPath -Recurse -Force
}
New-Item -ItemType Directory -Path $OutputPath -Force | Out-Null

# Build with Native AOT
Write-Host "[2/6] Building CLI with Native AOT..." -ForegroundColor Green
$PublishPath = Join-Path $ProjectRoot "WildfireBuffet.CLI\bin\Release\net8.0\win-x64\publish"

Push-Location (Join-Path $ProjectRoot "WildfireBuffet.CLI")
try {
    dotnet publish -c Release -r win-x64 -p:PublishAot=true
    if ($LASTEXITCODE -ne 0) {
        Write-Host "Build failed!" -ForegroundColor Red
        exit 1
    }
}
finally {
    Pop-Location
}

# Create package structure
Write-Host "[3/6] Creating package structure..." -ForegroundColor Green
$PackageRoot = Join-Path $OutputPath "WildfireBuffet-CLI"
New-Item -ItemType Directory -Path $PackageRoot -Force | Out-Null

# Copy executable
Write-Host "[4/6] Copying files..." -ForegroundColor Green
Copy-Item -Path (Join-Path $PublishPath "WildfireBuffet.CLI.exe") -Destination $PackageRoot

# Copy directories
$directories = @("Modules", "ExecutionTypes", "Packagers", "Recipes", "Templates", "Shellcode")
foreach ($dir in $directories) {
    $srcPath = Join-Path $ProjectRoot $dir
    if (Test-Path $srcPath) {
        Write-Host "  Copying $dir..." -ForegroundColor Gray
        Copy-Item -Path $srcPath -Destination (Join-Path $PackageRoot $dir) -Recurse
    }
}

# Create output directory
New-Item -ItemType Directory -Path (Join-Path $PackageRoot "output") -Force | Out-Null

# Create config file
Write-Host "[5/6] Creating configuration..." -ForegroundColor Green
$config = @{
    modulesDirectory = "Modules"
    executionTypesDirectory = "ExecutionTypes"
    packagersDirectory = "Packagers"
    recipesDirectory = "Recipes"
    templatesDirectory = "Templates"
    outputDirectory = "output"
}
$config | ConvertTo-Json | Out-File -FilePath (Join-Path $PackageRoot "wildfire-config.json") -Encoding UTF8

# Create README
$readme = "# Wildfire Buffet CLI`n`nNative AOT compiled security testing payload generator.`n`n## Usage`n`nInteractive mode:`n  WildfireBuffet.CLI.exe`n`nBuild from recipe:`n  WildfireBuffet.CLI.exe --config Recipes\your-recipe.json`n`n## Author`nril3y"
$readme | Out-File -FilePath (Join-Path $PackageRoot "README.txt") -Encoding UTF8

# Create ZIP
Write-Host "[6/6] Creating ZIP package..." -ForegroundColor Green
$ZipPath = Join-Path $OutputPath "WildfireBuffet-CLI-win-x64.zip"
Add-Type -AssemblyName System.IO.Compression.FileSystem
[System.IO.Compression.ZipFile]::CreateFromDirectory($PackageRoot, $ZipPath)

$ZipSize = (Get-Item $ZipPath).Length / 1MB
Write-Host ""
Write-Host "========================================" -ForegroundColor Green
Write-Host "Package created successfully!" -ForegroundColor Green
Write-Host "========================================" -ForegroundColor Green
Write-Host "Location: $ZipPath" -ForegroundColor Cyan
$sizeText = "Size: {0:N2} MB" -f $ZipSize
Write-Host $sizeText -ForegroundColor Cyan
Write-Host ""
