# Package Wildfire Buffet CLI using Docker (avoids local toolchain issues)
param(
    [string]$OutputPath = ".\dist"
)

$ErrorActionPreference = "Stop"

Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Wildfire Buffet CLI Packaging (Docker)" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host ""

# Check Docker
Write-Host "[1/5] Checking Docker..." -ForegroundColor Yellow
try {
    docker --version | Out-Null
    Write-Host "  Docker is available" -ForegroundColor Green
}
catch {
    Write-Host "ERROR: Docker not found" -ForegroundColor Red
    Write-Host "Install Docker Desktop: https://www.docker.com/products/docker-desktop/" -ForegroundColor Cyan
    exit 1
}

# Build Docker image
Write-Host "[2/5] Building Docker image with Native AOT..." -ForegroundColor Green
Write-Host "  This will take several minutes..." -ForegroundColor Gray
docker-compose build
if ($LASTEXITCODE -ne 0) {
    Write-Host "Docker build failed!" -ForegroundColor Red
    exit 1
}

# Setup paths
$ProjectRoot = Split-Path -Parent (Split-Path -Parent $PSCommandPath)
$OutputPath = Join-Path $ProjectRoot $OutputPath
if (Test-Path $OutputPath) {
    Remove-Item -Path $OutputPath -Recurse -Force
}
New-Item -ItemType Directory -Path $OutputPath -Force | Out-Null
$PackageRoot = Join-Path $OutputPath "WildfireBuffet-CLI"
New-Item -ItemType Directory -Path $PackageRoot -Force | Out-Null

# Extract files from Docker image
Write-Host "[3/5] Extracting files from Docker image..." -ForegroundColor Green

# Create temporary container
$containerId = docker create wildfire-buffet-cli:latest
Write-Host "  Created temporary container: $containerId" -ForegroundColor Gray

try {
    # Copy executable
    Write-Host "  Copying executable..." -ForegroundColor Gray
    docker cp "$containerId`:C:\wildfire-buffet\WildfireBuffet.CLI.exe" $PackageRoot

    # Copy directories
    $directories = @("Modules", "ExecutionTypes", "Packagers", "Recipes", "Templates", "Shellcode")
    foreach ($dir in $directories) {
        Write-Host "  Copying $dir..." -ForegroundColor Gray
        docker cp "$containerId`:C:\wildfire-buffet\$dir" $PackageRoot
    }

    # Copy config
    docker cp "$containerId`:C:\wildfire-buffet\wildfire-config.json" $PackageRoot
}
finally {
    # Cleanup temp container
    docker rm $containerId | Out-Null
}

# Create output directory
New-Item -ItemType Directory -Path (Join-Path $PackageRoot "output") -Force | Out-Null

# Create README
Write-Host "[4/5] Creating documentation..." -ForegroundColor Green
$readme = "# Wildfire Buffet CLI`n`nNative AOT compiled security testing payload generator.`n`n## Usage`n`nInteractive mode:`n  WildfireBuffet.CLI.exe`n`nBuild from recipe:`n  WildfireBuffet.CLI.exe --config Recipes\your-recipe.json`n`n## Author`nril3y"
$readme | Out-File -FilePath (Join-Path $PackageRoot "README.txt") -Encoding UTF8

# Create ZIP
Write-Host "[5/5] Creating ZIP package..." -ForegroundColor Green
$ZipPath = Join-Path $OutputPath "WildfireBuffet-CLI-win-x64.zip"
Add-Type -AssemblyName System.IO.Compression.FileSystem
[System.IO.Compression.ZipFile]::CreateFromDirectory($PackageRoot, $ZipPath)

$ZipSize = (Get-Item $ZipPath).Length / 1MB
Write-Host ""
Write-Host "========================================" -ForegroundColor Green
Write-Host "Package created successfully!" -ForegroundColor Green
Write-Host "========================================" -ForegroundColor Green
Write-Host "Location: $ZipPath" -ForegroundColor Cyan
$sizeText = "Size: {0:N2} MB" -f $ZipSize
Write-Host $sizeText -ForegroundColor Cyan
Write-Host ""
Write-Host "Ready for distribution to testing team!" -ForegroundColor Green
Write-Host ""
