#!/usr/bin/env pwsh
<#
.SYNOPSIS
    Builds and packages Wildfire Buffet CLI and UI for release distribution.

.DESCRIPTION
    This script:
    1. Builds CLI (Native AOT) and UI (Self-contained) for Windows x64
    2. Packages each with all necessary folders (Modules, ExecutionTypes, etc.)
    3. Creates ZIP files ready for distribution
    4. Each package is standalone - just extract and run

.PARAMETER BuildCLI
    Build and package the CLI

.PARAMETER BuildUI
    Build and package the UI

.PARAMETER OutputDir
    Directory where packages will be created (default: ./packages)

.PARAMETER Clean
    Clean build artifacts before building

.EXAMPLE
    .\Scripts\Build-Release.ps1 -BuildCLI -BuildUI
    Builds both CLI and UI packages

.EXAMPLE
    .\Scripts\Build-Release.ps1 -BuildCLI -Clean
    Cleans and builds only CLI package
#>

[CmdletBinding()]
param(
    [switch]$BuildCLI = $false,
    [switch]$BuildUI = $false,
    [string]$OutputDir = "packages",
    [switch]$Clean = $false
)

$ErrorActionPreference = "Stop"

# Get script directory and project root
$ScriptDir = Split-Path -Parent $PSCommandPath
$ProjectRoot = Split-Path -Parent $ScriptDir

Write-Host "================================" -ForegroundColor Cyan
Write-Host "Wildfire Buffet Release Builder" -ForegroundColor Cyan
Write-Host "================================" -ForegroundColor Cyan
Write-Host ""

# If no flags specified, build both
if (-not $BuildCLI -and -not $BuildUI) {
    Write-Host "No build target specified, building both CLI and UI..." -ForegroundColor Yellow
    $BuildCLI = $true
    $BuildUI = $true
}

# Create output directory
$PackagesDir = Join-Path $ProjectRoot $OutputDir
if (-not (Test-Path $PackagesDir)) {
    New-Item -ItemType Directory -Path $PackagesDir -Force | Out-Null
}

# Clean if requested
if ($Clean) {
    Write-Host "[Clean] Removing build artifacts..." -ForegroundColor Yellow
    $ArtifactsDir = Join-Path $ProjectRoot "artifacts"
    if (Test-Path $ArtifactsDir) {
        Remove-Item -Recurse -Force $ArtifactsDir
    }
    dotnet clean $ProjectRoot
    Write-Host "[Clean] Done" -ForegroundColor Green
    Write-Host ""
}

# Function to copy necessary folders
function Copy-RuntimeFolders {
    param(
        [string]$DestinationPath
    )

    Write-Host "  Copying runtime folders..." -ForegroundColor Gray

    $FoldersToCopy = @(
        "Modules",
        "ExecutionTypes",
        "Packagers",
        "Recipes",
        "Templates",
        "Shellcode",
        "Scripts"
    )

    foreach ($folder in $FoldersToCopy) {
        $SourcePath = Join-Path $ProjectRoot $folder
        if (Test-Path $SourcePath) {
            $DestFolder = Join-Path $DestinationPath $folder
            Write-Host "    - $folder" -ForegroundColor DarkGray
            Copy-Item -Path $SourcePath -Destination $DestFolder -Recurse -Force
        }
    }

    # Copy config and docs
    $FilesToCopy = @(
        "wildfire-config.json",
        "README.md",
        "LICENSE",
        "DOCKER.md"
    )

    foreach ($file in $FilesToCopy) {
        $SourceFile = Join-Path $ProjectRoot $file
        if (Test-Path $SourceFile) {
            Write-Host "    - $file" -ForegroundColor DarkGray
            Copy-Item -Path $SourceFile -Destination $DestinationPath -Force
        }
    }
}

# Function to create ZIP
function Create-ZipArchive {
    param(
        [string]$SourcePath,
        [string]$ZipPath
    )

    Write-Host "  Creating ZIP archive..." -ForegroundColor Gray

    # Remove existing ZIP if present
    if (Test-Path $ZipPath) {
        Remove-Item $ZipPath -Force
    }

    # Create ZIP using .NET
    Add-Type -AssemblyName System.IO.Compression.FileSystem
    [System.IO.Compression.ZipFile]::CreateFromDirectory($SourcePath, $ZipPath)

    $ZipSize = (Get-Item $ZipPath).Length / 1MB
    Write-Host "  ZIP created: $([System.IO.Path]::GetFileName($ZipPath)) ($("{0:N2}" -f $ZipSize) MB)" -ForegroundColor Green
}

# Build CLI
if ($BuildCLI) {
    Write-Host "[CLI] Building CLI (Native AOT - Windows x64)..." -ForegroundColor Cyan

    $CLIProject = Join-Path $ProjectRoot "WildfireBuffet.CLI\WildfireBuffet.CLI.csproj"
    $CLIArtifacts = Join-Path $ProjectRoot "artifacts\cli-win-x64"
    $CLIPackage = Join-Path $ProjectRoot "artifacts\wildfire-buffet-cli-win-x64"

    # Build
    Write-Host "  Publishing..." -ForegroundColor Gray
    dotnet publish $CLIProject -c Release -r win-x64 -p:PublishAot=true -o $CLIArtifacts

    if ($LASTEXITCODE -ne 0) {
        Write-Host "[CLI] Build failed!" -ForegroundColor Red
        exit 1
    }

    Write-Host "  Build successful!" -ForegroundColor Green

    # Create package directory
    Write-Host "  Creating package..." -ForegroundColor Gray
    if (Test-Path $CLIPackage) {
        Remove-Item -Recurse -Force $CLIPackage
    }
    New-Item -ItemType Directory -Path $CLIPackage -Force | Out-Null

    # Copy executable
    Write-Host "  Copying executable..." -ForegroundColor Gray
    Copy-Item -Path (Join-Path $CLIArtifacts "*") -Destination $CLIPackage -Recurse -Force

    # Copy folders
    Copy-RuntimeFolders -DestinationPath $CLIPackage

    # Create ZIP
    $CLIZip = Join-Path $PackagesDir "wildfire-buffet-cli-win-x64.zip"
    Create-ZipArchive -SourcePath $CLIPackage -ZipPath $CLIZip

    Write-Host "[CLI] Package complete!" -ForegroundColor Green
    Write-Host ""
}

# Build UI
if ($BuildUI) {
    Write-Host "[UI] Building UI (Self-contained - Windows x64)..." -ForegroundColor Cyan

    $UIProject = Join-Path $ProjectRoot "WildfireBuffet.UI\WildfireBuffet.UI.csproj"
    $UIArtifacts = Join-Path $ProjectRoot "artifacts\ui-win-x64"
    $UIPackage = Join-Path $ProjectRoot "artifacts\wildfire-buffet-ui-win-x64"

    # Build
    Write-Host "  Publishing..." -ForegroundColor Gray
    dotnet publish $UIProject -c Release -r win-x64 --self-contained -o $UIArtifacts

    if ($LASTEXITCODE -ne 0) {
        Write-Host "[UI] Build failed!" -ForegroundColor Red
        exit 1
    }

    Write-Host "  Build successful!" -ForegroundColor Green

    # Create package directory
    Write-Host "  Creating package..." -ForegroundColor Gray
    if (Test-Path $UIPackage) {
        Remove-Item -Recurse -Force $UIPackage
    }
    New-Item -ItemType Directory -Path $UIPackage -Force | Out-Null

    # Copy executable and dependencies
    Write-Host "  Copying application..." -ForegroundColor Gray
    Copy-Item -Path (Join-Path $UIArtifacts "*") -Destination $UIPackage -Recurse -Force

    # Copy folders
    Copy-RuntimeFolders -DestinationPath $UIPackage

    # Create ZIP
    $UIZip = Join-Path $PackagesDir "wildfire-buffet-ui-win-x64.zip"
    Create-ZipArchive -SourcePath $UIPackage -ZipPath $UIZip

    Write-Host "[UI] Package complete!" -ForegroundColor Green
    Write-Host ""
}

Write-Host "================================" -ForegroundColor Cyan
Write-Host "Build Complete!" -ForegroundColor Green
Write-Host "Packages location: $PackagesDir" -ForegroundColor Cyan
Write-Host ""

# List created packages
Write-Host "Created packages:" -ForegroundColor Cyan
Get-ChildItem -Path $PackagesDir -Filter "*.zip" | ForEach-Object {
    $Size = $_.Length / 1MB
    Write-Host "  - $($_.Name) ($("{0:N2}" -f $Size) MB)" -ForegroundColor White
}

Write-Host ""
Write-Host "Extract and run! Each package is standalone." -ForegroundColor Green
Write-Host "================================" -ForegroundColor Cyan
