# Build all recipes to unified output directory
# Output: X:\Projects\dotnet\wildfire-buffet\output

param(
    [string]$CLI = "X:\Projects\dotnet\wildfire-buffet\WildfireBuffet.CLI\bin\Debug\net8.0\WildfireBuffet.CLI.exe",
    [string]$RecipesDir = "X:\Projects\dotnet\wildfire-buffet\Recipes",
    [string]$OutputDir = "X:\Projects\dotnet\wildfire-buffet\output"
)

Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Building All Recipes" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "CLI: $CLI"
Write-Host "Recipes: $RecipesDir"
Write-Host "Output: $OutputDir"
Write-Host ""

# Get all recipe files
$recipes = Get-ChildItem -Path $RecipesDir -Filter "*.json" | Sort-Object Name

Write-Host "Found $($recipes.Count) recipes" -ForegroundColor Green
Write-Host ""

$results = @()

foreach ($recipe in $recipes) {
    Write-Host "[$($recipes.IndexOf($recipe) + 1)/$($recipes.Count)] Building: $($recipe.Name)" -ForegroundColor Yellow

    $startTime = Get-Date

    try {
        # Build the recipe
        $output = & $CLI --config $recipe.FullName 2>&1
        $exitCode = $LASTEXITCODE

        $elapsed = ((Get-Date) - $startTime).TotalSeconds

        # Check for success
        if ($output -match '(\d+) succeeded') {
            $successCount = $matches[1]
            Write-Host "  Success: $successCount sample(s) in $([math]::Round($elapsed, 1))s" -ForegroundColor Green

            $results += [PSCustomObject]@{
                Recipe = $recipe.Name
                Status = 'Success'
                Samples = $successCount
                Time = "$([math]::Round($elapsed, 1))s"
            }
        }
        elseif ($output -match '(\d+) failed') {
            $failCount = $matches[1]
            Write-Host "  Failed: $failCount sample(s)" -ForegroundColor Red

            # Show error details
            if ($output -match 'Exception|Error:') {
                $errorLines = $output | Select-String -Pattern 'Exception|Error:' | Select-Object -First 2
                foreach ($line in $errorLines) {
                    Write-Host "    $line" -ForegroundColor Red
                }
            }

            $results += [PSCustomObject]@{
                Recipe = $recipe.Name
                Status = 'Failed'
                Samples = 0
                Time = "$([math]::Round($elapsed, 1))s"
            }
        }
        else {
            Write-Host "  Unknown result" -ForegroundColor Yellow
            $results += [PSCustomObject]@{
                Recipe = $recipe.Name
                Status = 'Unknown'
                Samples = 0
                Time = "$([math]::Round($elapsed, 1))s"
            }
        }
    }
    catch {
        Write-Host "  Error: $_" -ForegroundColor Red
        $results += [PSCustomObject]@{
            Recipe = $recipe.Name
            Status = 'Error'
            Samples = 0
            Time = 'N/A'
        }
    }

    Write-Host ""
}

Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Summary" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host ""

$results | Format-Table -AutoSize

$successCount = ($results | Where-Object { $_.Status -eq 'Success' }).Count
$failCount = ($results | Where-Object { $_.Status -eq 'Failed' -or $_.Status -eq 'Error' }).Count
$totalSamples = ($results | Where-Object { $_.Status -eq 'Success' } | Measure-Object -Property Samples -Sum).Sum

Write-Host "Total Recipes: $($recipes.Count)" -ForegroundColor Cyan
Write-Host "Succeeded: $successCount" -ForegroundColor Green
Write-Host "Failed: $failCount" -ForegroundColor $(if ($failCount -gt 0) { "Red" } else { "Gray" })
Write-Host "Total Samples Generated: $totalSamples" -ForegroundColor Green
Write-Host ""

# List output files
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Output Directory Contents" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Directory: $OutputDir"
Write-Host ""

if (Test-Path $OutputDir) {
    $outputFiles = Get-ChildItem -Path $OutputDir -File | Sort-Object LastWriteTime -Descending

    if ($outputFiles.Count -gt 0) {
        $outputFiles | Select-Object Name, @{N='Size';E={'{0:N2} MB' -f ($_.Length / 1MB)}}, LastWriteTime | Format-Table -AutoSize
        Write-Host "Total Files: $($outputFiles.Count)" -ForegroundColor Green

        $totalSize = ($outputFiles | Measure-Object -Property Length -Sum).Sum / 1MB
        Write-Host "Total Size: $([math]::Round($totalSize, 2)) MB" -ForegroundColor Green
    }
    else {
        Write-Host "No files found in output directory" -ForegroundColor Yellow
    }
}
else {
    Write-Host "Output directory does not exist" -ForegroundColor Red
}
