# Wildfire Buffet

A modular .NET security testing framework for generating polymorphic test executables with configurable evasion techniques and behaviors.

**Author:** ril3y

## Developer Guide

For detailed information on building, extending, and testing Wildfire Buffet, see the [Developer Guide](DEVELOPER.md).

## Overview

Wildfire Buffet is designed for security professionals to test their product's detection capabilities. It generates unique executable samples with various evasion techniques and behaviors, ensuring each build produces a different hash through polymorphic code generation.

The framework features a unique **hot-reload architecture** - drop `.cs` module files into directories and they're immediately available without recompiling the framework. It includes both an **interactive CLI** (Spectre.Console) and a **cross-platform GUI** (Avalonia 11), plus support for **automation via JSON recipes**.

## Key Features

✅ **Hot-Reload Architecture** - Drop `.cs` files, no framework recompilation needed!
✅ **Multiple Execution Types** - Native EXE, PDF, LNK, PowerShell, Excel formats
✅ **Polymorphic Generation** - Each sample has a unique hash
✅ **Cross-Platform UI** - Avalonia 11 UI works on Windows, Linux, macOS
✅ **Interactive CLI** - Beautiful terminal UI powered by Spectre.Console
✅ **Command-Line Mode** - Automate builds with JSON configurations
✅ **Configurable Parameters** - Rich parameter types including dropdowns
✅ **Dynamic Compilation** - Uses Roslyn and `dotnet publish`
✅ **Comprehensive Test Framework** - 75 tests at 100% pass rate
✅ **Packaging System** - ZIP, padding, and custom packagers

## Screenshots

### Interactive CLI
![CLI Interface](screenshots/cli.png)

The Spectre.Console-powered CLI provides a beautiful terminal experience with:
- Interactive module selection with multi-select prompts
- Real-time build progress with spinners and progress bars
- Syntax-highlighted output with clickable links
- Detailed build results with file hashes and sizes

## Quick Start

There are multiple ways to use Wildfire Buffet - choose the one that fits your workflow:

### Option 1: Interactive CLI (Recommended for Beginners)

```bash
# Clone and navigate to the project
cd wildfire-buffet

# Run the interactive CLI (no build needed!)
cd WildfireBuffet.CLI
dotnet run

# Follow the interactive prompts to:
# 1. Select modules (evasions and actions)
# 2. Configure parameters
# 3. Set output options
# 4. Generate payloads
```

### Option 2: Recipe-Based Automation (Recommended for Repeated Builds)

```bash
# Use a pre-made recipe file
dotnet run --project WildfireBuffet.CLI -- --config Recipes/native-exe-evasive.json

# Or after building:
WildfireBuffet.CLI.exe --config Recipes/pdf-invoice-phishing.json
```

### Option 3: Cross-Platform GUI (Avalonia UI)

```bash
# Run the GUI application
cd WildfireBuffet.UI
dotnet run

# Or build and run the standalone executable
dotnet build -c Release
cd bin/Release/net8.0
./WildfireBuffet.UI.exe
```

### Option 4: Docker (Isolated Environment)

```powershell
# Build the Docker image (Windows containers required)
docker-compose build

# Run with a recipe
docker-compose run --rm wildfire-buffet-cli --config Recipes/native-exe-evasive.json

# Interactive mode
docker-compose run --rm wildfire-buffet-cli --interactive

# See DOCKER.md for detailed Docker setup instructions
```

### Option 5: Portable Release Packages (Distribution)

```powershell
# Build standalone release packages (CLI and/or UI)
.\Scripts\Build-Release.ps1

# Or build both with clean build
.\Scripts\Build-Release.ps1 -Clean

# Or build only CLI
.\Scripts\Build-Release.ps1 -BuildCLI

# Or build only UI
.\Scripts\Build-Release.ps1 -BuildUI

# Creates packages in packages/ directory:
# - wildfire-buffet-cli-win-x64.zip (~15-20 MB)
# - wildfire-buffet-ui-win-x64.zip (~80-100 MB)
# Each package includes all necessary folders and configs
# Extract and run - no dependencies needed!
```

## Configuration System

Wildfire Buffet uses two types of configuration files:

### 1. Framework Configuration (`wildfire-config.json`)

This file tells the framework where to find modules, recipes, and other resources:

```json
{
  "modulesDirectory": "Modules",
  "executionTypesDirectory": "ExecutionTypes",
  "packagersDirectory": "Packagers",
  "recipesDirectory": "Recipes",
  "templatesDirectory": "Templates",
  "outputDirectory": "output"
}
```

**Setup:**
1. Copy `wildfire-config.example.json` to `wildfire-config.json`
2. Customize paths if needed (defaults work for most users)
3. All paths are relative to the working directory

### 2. Recipe Files (`Recipes/*.json`)

Recipes define what to build - they're like build templates you can reuse:

**Example:** `Recipes/native-exe-evasive.json`
```json
{
  "name": "Evasive Native Executable",
  "description": "Native .exe with VM detection and evasion checks",
  "icon": "🔐",
  "executionTypeId": "native-executable",
  "modules": [
    {
      "id": "cpu-core-check",
      "parameters": {
        "MinCores": 2
      }
    },
    {
      "id": "ram-size-check",
      "parameters": {
        "MinRamGB": 4
      }
    },
    {
      "id": "hard-disk-check",
      "parameters": {
        "MinDiskSizeGB": 80
      }
    }
  ],
  "sampleCount": 5,
  "randomizationLevel": 7,
  "packagerId": "zip-archive"
}
```

**Available Recipes:**
- `native-exe-evasive.json` - Native executable with anti-VM checks
- `native-exe-dynamic-import.json` - Dynamic API resolution demo
- `native-exe-process-injection-*.json` - Process injection examples
- `pdf-invoice-phishing.json` - Phishing PDF with links
- `pdf-advanced-with-params.json` - Complex PDF with all features
- `lnk-credential-harvester.json` - LNK shortcut payload
- `excel-macro-downloader.json` - Excel macro example
- `powershell-beacon.json` - PowerShell script beacon

**Creating Your Own Recipe:**
1. Copy an existing recipe from `Recipes/`
2. Modify the `executionTypeId`, `modules`, and `parameters`
3. Save with a descriptive name
4. Run: `WildfireBuffet.CLI.exe --config Recipes/your-recipe.json`

**Tip:** Use interactive mode, then save the configuration when prompted. This generates a recipe file you can reuse!

## Project Structure

```
WildfireBuffet/
├── Modules/                       # 🔥 DROP YOUR MODULE .CS FILES HERE!
│   ├── Evasions/                  # VM/Sandbox detection (4 modules)
│   │   ├── CpuCoreCheck.cs
│   │   ├── RamSizeCheck.cs
│   │   ├── HardDiskCheck.cs
│   │   └── DynamicImportResolver.cs
│   └── Actions/                   # Runtime behaviors (14 modules)
│       ├── UrlVisitor.cs          # General: HTTP requests
│       ├── RegistryOps.cs         # General: Registry operations
│       ├── ProcessInjection.cs    # General: Process injection
│       ├── CommandExecutor.cs     # General: Shell commands
│       ├── PdfText.cs             # PDF: Text content
│       ├── PdfHyperlink.cs        # PDF: Clickable links
│       ├── PdfJavaScript.cs       # PDF: JS injection
│       ├── PdfImage.cs            # PDF: Image embedding
│       ├── LnkCommand.cs          # LNK: Command execution
│       ├── LnkUrl.cs              # LNK: URL shortcuts
│       ├── PsWebRequest.cs        # PowerShell: Web requests
│       ├── PsFileOperation.cs     # PowerShell: File ops
│       └── ExcelWebRequest.cs     # Excel: VBA web requests
│
├── ExecutionTypes/                # 🔥 DROP EXECUTION TYPE .CS FILES HERE!
│   ├── NativeExecutable.cs        # Self-contained .NET (33MB)
│   ├── NativeExecutableAOT.cs     # NativeAOT binary (8MB)
│   ├── PdfDocument.cs             # PDF with links/JS
│   ├── LnkShortcut.cs             # Windows .lnk files
│   ├── PowerShellScript.cs        # PowerShell .ps1 scripts
│   └── ExcelMacro.cs              # Excel .xlsm with VBA
│
├── Packagers/                     # 🔥 DROP PACKAGER .CS FILES HERE!
│   ├── ZipArchivePackager.cs      # ZIP compression
│   └── FilePaddingPackager.cs     # File size padding
│
├── WildfireBuffet.Core/           # Core framework
│   ├── Abstractions/              # Interfaces & base classes
│   │   ├── IExecutionType.cs
│   │   ├── ExecutionTypeBase.cs
│   │   ├── IPackager.cs
│   │   └── PackagerBase.cs
│   ├── Builder/                   # Build orchestration
│   │   ├── PayloadBuilder.cs      # Main build pipeline
│   │   ├── BuildConfiguration.cs  # Config models
│   │   └── ConfigurationManager.cs
│   ├── Discovery/                 # Hot-reload providers
│   │   ├── FileBasedModuleLoader.cs
│   │   ├── ExecutionTypeProvider.cs
│   │   ├── PackagerProvider.cs
│   │   └── ModuleFileParser.cs
│   ├── Engine/                    # Compilation & polymorphism
│   │   └── PolymorphicEngine.cs
│   └── Models/                    # Data models
│       ├── ModuleDefinition.cs
│       ├── ModuleParameter.cs
│       ├── BuildResult.cs
│       └── ExecutionContext.cs
│
├── WildfireBuffet.UI/             # Avalonia 11 cross-platform UI
│   ├── ViewModels/                # MVVM view models
│   ├── Views/                     # XAML views
│   └── Assets/                    # UI resources
│
├── WildfireBuffet.CLI/            # Spectre.Console terminal UI
│   └── Program.cs                 # Interactive & CLI mode
│
├── WildfireBuffet.Tests/          # Test framework (75 tests)
│   ├── TestFixture.cs             # Shared xUnit fixture
│   ├── TestBase.cs                # Base test class
│   ├── Configuration/             # Config tests
│   ├── ExecutionTypes/            # Execution type tests
│   ├── Modules/                   # Module tests
│   ├── Packagers/                 # Packager tests
│   └── TestData/                  # Helper data
│
└── README.md
```

## Building and Packaging

### Development Build

```bash
# Build all projects
dotnet build

# Build in Release mode
dotnet build -c Release

# Run tests
dotnet test
```

### Release Packages (Recommended for Distribution)

**Quick Start - Build Everything:**
```powershell
# Build both CLI and UI packages (default)
.\Scripts\Build-Release.ps1

# Or with clean build (recommended)
.\Scripts\Build-Release.ps1 -Clean
```

**What You Get:**
- `packages/wildfire-buffet-cli-win-x64.zip` (~15-20 MB)
  - Native AOT executable (single .exe, no runtime needed)
  - All modules, execution types, packagers
  - Recipes, templates, shellcode
  - Configuration files
  - Ready to extract and run!

- `packages/wildfire-buffet-ui-win-x64.zip` (~80-100 MB)
  - Self-contained UI application
  - All modules, execution types, packagers
  - Recipes, templates, shellcode
  - Configuration files
  - Ready to extract and run!

**Build Options:**
```powershell
# Build only CLI
.\Scripts\Build-Release.ps1 -BuildCLI

# Build only UI
.\Scripts\Build-Release.ps1 -BuildUI

# Build both with clean (removes old artifacts first)
.\Scripts\Build-Release.ps1 -Clean

# Specify custom output directory
.\Scripts\Build-Release.ps1 -OutputDir "releases"
```

**Package Structure:**
```
wildfire-buffet-cli-win-x64/
├── WildfireBuffet.CLI.exe      # Single executable (Native AOT)
├── Modules/                     # All module .cs files
│   ├── Evasions/
│   └── Actions/
├── ExecutionTypes/              # Execution type definitions
├── Packagers/                   # Packager implementations
├── Recipes/                     # Recipe JSON files
├── Templates/                   # Code templates
├── Shellcode/                   # Shellcode files (if present)
├── Scripts/                     # Helper scripts
├── wildfire-config.json         # Configuration (relative paths)
├── README.md                    # Documentation
└── LICENSE                      # License file
```

### Manual Build Options

**CLI - Native AOT (Smallest, Fastest)**
```bash
cd WildfireBuffet.CLI
dotnet publish -c Release -r win-x64 -p:PublishAot=true
# Output: ~8MB native executable with no .NET runtime dependency
# Location: bin/Release/net8.0/win-x64/publish/WildfireBuffet.CLI.exe
```

**CLI - Self-Contained (No Dependencies)**
```bash
cd WildfireBuffet.CLI
dotnet publish -c Release -r win-x64 --self-contained
# Output: ~35MB with bundled .NET runtime
# Location: bin/Release/net8.0/win-x64/publish/WildfireBuffet.CLI.exe
```

**UI - Self-Contained**
```bash
cd WildfireBuffet.UI
dotnet publish -c Release -r win-x64 --self-contained
# Output: ~80-100MB (includes Avalonia UI framework)
# Location: bin/Release/net8.0/win-x64/publish/WildfireBuffet.UI.exe
```

**Note:** Manual builds only create the executable. For distribution, use `Build-Release.ps1` to package with all necessary folders and configurations.

### Docker Build

```powershell
# Build Docker image with Native AOT compilation
docker build -t wildfire-buffet-cli:latest .

# Or use docker-compose
docker-compose build

# Run
docker run --rm -v ${PWD}/output:/wildfire-buffet/output wildfire-buffet-cli:latest --config Recipes/native-exe-evasive.json
```

See [DOCKER.md](DOCKER.md) for detailed Docker instructions.

### Build Requirements

**Standard Build:**
- .NET 8 SDK
- Windows, Linux, or macOS

**Native AOT Build (Optional):**
- .NET 8 SDK
- Visual Studio 2022 with "Desktop development with C++" workload
- OR Visual Studio Build Tools 2022 with C++ components
- ~7GB disk space for build tools

**Installing Native AOT Dependencies:**

Native AOT compilation requires C++ build tools. Choose one of these options:

**Option 1: Visual Studio 2022 (Full IDE)**
1. Download Visual Studio 2022 from: https://visualstudio.microsoft.com/downloads/
2. Run the installer
3. Select "Desktop development with C++" workload
4. Ensure these components are selected:
   - MSVC v143 - VS 2022 C++ x64/x86 build tools
   - Windows 10/11 SDK (latest version)
   - C++ CMake tools for Windows
5. Complete installation (~7GB disk space)

**Option 2: Visual Studio Build Tools 2022 (Command-Line Only - Recommended for CI/CD)**
1. Download Build Tools from: https://visualstudio.microsoft.com/downloads/#build-tools-for-visual-studio-2022
2. Run vs_buildtools.exe
3. Select "Desktop development with C++" workload
4. Ensure these components are selected:
   - MSVC v143 - VS 2022 C++ x64/x86 build tools
   - Windows 10/11 SDK (latest version)
5. Complete installation (~3-4GB disk space)

**Verifying Installation:**
```powershell
# Find Visual Studio installation path
"C:\Program Files (x86)\Microsoft Visual Studio\Installer\vswhere.exe" -latest -requires Microsoft.VisualStudio.Component.VC.Tools.x86.x64 -property installationPath

# Should output something like:
# C:\Program Files\Microsoft Visual Studio\2022\Community
# or
# C:\Program Files (x86)\Microsoft Visual Studio\2022\BuildTools
```

**Alternative: Using Winget (Windows Package Manager)**
```powershell
# Install Visual Studio 2022 Community with C++ workload
winget install Microsoft.VisualStudio.2022.Community --override "--add Microsoft.VisualStudio.Workload.NativeDesktop --includeRecommended --passive"

# Or install Build Tools only
winget install Microsoft.VisualStudio.2022.BuildTools --override "--add Microsoft.VisualStudio.Workload.VCTools --includeRecommended --passive"
```

**Docker Build:**
- Docker Desktop with Windows containers enabled
- Windows 10/11 Pro, Enterprise, or Education
- 20GB free disk space

## Available Execution Types (6)

### Native Executables
- **native-executable** - Self-contained .NET executable with bundled runtime (~33MB)
- **native-executable-aot** - Pure native binary via NativeAOT compilation (~8MB, faster startup)

### Document & Script Formats
- **pdf-document** - PDF files with clickable links, JavaScript, and images (via QuestPDF)
- **lnk-shortcut** - Windows .lnk shortcut files that execute commands or URLs
- **powershell-script** - PowerShell .ps1 scripts with polymorphic syntax variations
- **excel-macro** - Excel .xlsm files with VBA macros (via NPOI)

Each execution type supports polymorphic output generation - multiple samples from the same recipe will have unique hashes.

## Available Modules (18)

### Evasion Modules (4)
- **cpu-core-check** - Detects VMs by checking processor count
- **ram-size-check** - Exits if RAM is below threshold
- **hard-disk-check** - Detects VMs with small virtual disks
- **dynamic-import-resolver** - Hides imports using LoadLibrary/GetProcAddress

### Action Modules (14)

**General Actions:**
- **url-visitor** - Makes HTTP requests to specified URLs
- **registry-ops** - Create, read, or delete registry keys (dropdown parameter)
- **process-injection** - Injects shellcode into target processes
- **process-injection-functional** - Functional approach to process injection
- **command-executor** - Executes shell commands

**PDF-Specific Actions:**
- **pdf-text** - Adds text content to PDF documents
- **pdf-hyperlink** - Adds clickable hyperlinks to PDFs
- **pdf-image** - Embeds images in PDF documents
- **pdf-javascript** - Injects JavaScript code into PDFs

**LNK-Specific Actions:**
- **lnk-command** - Configures LNK shortcut to execute commands
- **lnk-url** - Configures LNK shortcut to open URLs

**PowerShell-Specific Actions:**
- **ps-web-request** - Makes web requests from PowerShell
- **ps-file-operation** - Performs file operations in PowerShell

**Excel-Specific Action:**
- **excel-web-request** - Performs web requests from VBA macro

## Creating Your Own Modules

### Module File Format

Create a `.cs` file with this structure:

```csharp
/*
 * MODULE: My Custom Check
 * ID: my-custom-check
 * DESCRIPTION: Does something awesome
 * CATEGORY: Hardware Detection
 *
 * PARAMETERS:
 * @MinValue:int:10:1:100:Minimum value required
 * @EnableFeature:bool:true:::Enable this feature
 * @CustomText:string:Hello:::Custom text to use
 *
 * FORMAT: @ParameterName:Type:DefaultValue:MinValue:MaxValue:Description
 */

// USINGS
using System;

// CODE
var myValue = Environment.GetSomeValue();
Console.WriteLine($"Detected value: {myValue}");

if (myValue < @MinValue)
{
    Console.WriteLine("Value too low - exiting");
    Environment.Exit(1);
}
```

### Adding a New Module

**1. Create the file:**
```bash
# Windows
notepad Modules\Evasions\MyCheck.cs

# Or just drop it in the folder
```

**2. Use the format above** with:
- Module metadata in comments
- Parameter definitions with `@ParameterName:type:default:min:max:description`
- Usings section
- Code section with `@ParameterName` placeholders

**3. Run the builder** - that's it! No recompilation needed!

```bash
dotnet run --project WildfireBuffet.CLI
```

### Parameter Types

- `int` - Integer values (with optional min/max)
- `bool` - True/false values
- `string` - Text values
- `stringlist` - Comma-separated list (e.g., "url1.com,url2.com")
- `choice` - Dropdown selection from predefined options (e.g., "Read|Write|Delete")

**Choice Parameter Example:**
```csharp
/*
 * PARAMETERS:
 * @Operation:choice:Read:Read|Write|Delete:::Operation to perform
 */
```

This creates a dropdown in the UI with options: Read, Write, Delete (default: Read)

### Example: Debugger Detection Module

```csharp
/*
 * MODULE: Debugger Detection
 * ID: debugger-check
 * DESCRIPTION: Exits if a debugger is attached
 * CATEGORY: Anti-Debug
 *
 * PARAMETERS:
 * (none)
 */

// USINGS
using System;
using System.Diagnostics;

// CODE
if (Debugger.IsAttached)
{
    Console.WriteLine("Debugger detected - exiting");
    Environment.Exit(1);
}
```

Save this as `Modules/Evasions/DebuggerCheck.cs` and it's ready to use!

## How It All Works Together

Understanding the system architecture helps you create effective payloads and extend the framework:

### The Build Pipeline

```
┌─────────────────────────────────────────────────────────────────┐
│                    1. USER INPUT                                 │
├─────────────────────────────────────────────────────────────────┤
│ • Select Execution Type (native-exe, PDF, LNK, PowerShell, etc.)│
│ • Select Modules (evasions + actions)                            │
│ • Configure Parameters                                           │
│ • Set sample count & randomization level                         │
└────────────────┬────────────────────────────────────────────────┘
                 │
                 ▼
┌─────────────────────────────────────────────────────────────────┐
│                    2. MODULE LOADING                             │
├─────────────────────────────────────────────────────────────────┤
│ FileBasedModuleLoader scans Modules/ directory:                 │
│ • Parses .cs files (ModuleFileParser)                            │
│ • Extracts metadata from comments (ID, name, description)        │
│ • Extracts parameters (@ParameterName:type:default)              │
│ • Extracts using statements and code blocks                      │
│ • Creates ModuleDefinition objects                               │
└────────────────┬────────────────────────────────────────────────┘
                 │
                 ▼
┌─────────────────────────────────────────────────────────────────┐
│                    3. CODE GENERATION                            │
├─────────────────────────────────────────────────────────────────┤
│ ExecutionType.GenerateAsync() is called:                        │
│ • Substitutes parameters (@MinRamGB → 4)                         │
│ • Combines module code in order                                  │
│ • Adds using statements                                          │
│ • Wraps in execution-type-specific structure                     │
│   - Native EXE: Main() with try/catch                            │
│   - PDF: QuestPDF document generation                            │
│   - LNK: Shortcut configuration                                  │
│   - PowerShell: Script with functions                            │
└────────────────┬────────────────────────────────────────────────┘
                 │
                 ▼
┌─────────────────────────────────────────────────────────────────┐
│                    4. POLYMORPHISM                               │
├─────────────────────────────────────────────────────────────────┤
│ PolymorphicEngine applies randomization:                        │
│ • Variable name obfuscation (var1 → temp_a3f2)                   │
│ • Junk code injection (meaningless but valid code)               │
│ • Random comments and whitespace                                 │
│ • Code block shuffling (where order doesn't matter)              │
│ • Random delays between operations                               │
│ Result: Each sample gets unique hash!                            │
└────────────────┬────────────────────────────────────────────────┘
                 │
                 ▼
┌─────────────────────────────────────────────────────────────────┐
│                    5. COMPILATION                                │
├─────────────────────────────────────────────────────────────────┤
│ For native executables:                                          │
│ • Create temp directory                                          │
│ • Write generated .cs file                                       │
│ • Create .csproj with dependencies                               │
│ • Run: dotnet publish -c Release -r win-x64                      │
│ • Collect output .exe                                            │
│                                                                  │
│ For documents (PDF/LNK):                                         │
│ • Use libraries (QuestPDF, Shell32) directly                     │
│ • Generate binary output                                         │
└────────────────┬────────────────────────────────────────────────┘
                 │
                 ▼
┌─────────────────────────────────────────────────────────────────┐
│                    6. PACKAGING (Optional)                       │
├─────────────────────────────────────────────────────────────────┤
│ If packager specified:                                           │
│ • ZipPackager: Creates ZIP archive with padding                  │
│ • IsoPackager: Creates ISO disk image                            │
│ • Custom packagers can add encryption, signing, etc.             │
└────────────────┬────────────────────────────────────────────────┘
                 │
                 ▼
┌─────────────────────────────────────────────────────────────────┐
│                    7. OUTPUT                                     │
├─────────────────────────────────────────────────────────────────┤
│ • Save to output directory                                       │
│ • Compute SHA256 hash                                            │
│ • Record metadata (size, timestamp, modules used)                │
│ • Display results to user                                        │
└─────────────────────────────────────────────────────────────────┘
```

### Component Interactions

**1. Modules (The Building Blocks)**
- Plain `.cs` files in `Modules/Evasions/` or `Modules/Actions/`
- Contain reusable code snippets with parameters
- Not compiled until used in a build
- Example: `RamSizeCheck.cs` checks if RAM > threshold

**2. Execution Types (The Output Format)**
- `.cs` files in `ExecutionTypes/` implementing `IExecutionType`
- Compiled dynamically at startup using Roslyn
- Define how modules are combined into final output
- Example: `NativeExecutable.cs` creates Windows .exe files

**3. Packagers (Post-Processing)**
- `.cs` files in `Packagers/` implementing `IPackager`
- Compiled dynamically at startup
- Process the final output (compression, padding, etc.)
- Example: `ZipPackager.cs` creates ZIP archives with optional padding

**4. Recipes (Build Templates)**
- JSON files in `Recipes/` directory
- Define complete build configurations
- Specify execution type, modules, parameters
- Reusable and shareable

### Example: Building a Native EXE with Evasions

```
1. User selects:
   - Execution Type: native-executable
   - Modules: [cpu-core-check, ram-size-check, url-visitor]
   - Parameters: MinCores=4, MinRamGB=8, Url="https://example.com"

2. FileBasedModuleLoader provides module code:
   - cpu-core-check → C# code checking Environment.ProcessorCount
   - ram-size-check → C# code checking GC.GetGCMemoryInfo()
   - url-visitor → C# code using HttpClient

3. NativeExecutable.GenerateAsync():
   - Builds Program.cs with:
     using System;
     using System.Net.Http;

     // CPU check code (with MinCores=4 substituted)
     if (Environment.ProcessorCount < 4) { Environment.Exit(1); }

     // RAM check code (with MinRamGB=8 substituted)
     var ram = GC.GetGCMemoryInfo().TotalAvailableMemoryBytes / (1024*1024*1024);
     if (ram < 8) { Environment.Exit(1); }

     // URL visitor code (with Url="https://example.com" substituted)
     using var client = new HttpClient();
     await client.GetAsync("https://example.com");

4. PolymorphicEngine adds randomization:
   - Renames variables
   - Adds junk code
   - Randomizes whitespace
   Result: Each build produces different binary hash!

5. CompilationEngine:
   - Creates temp .csproj
   - Runs: dotnet publish -r win-x64
   - Returns compiled payload.exe

6. Output saved to: output/payload_0001.exe
```

### Module Compatibility

Modules declare which execution types they support:

```csharp
/*
 * SUPPORTED_EXECUTION_TYPES: native-executable,native-executable-aot
 */
```

- `*` = Compatible with all execution types
- Specific IDs = Only works with those types
- The UI/CLI will only show compatible modules for selected execution type

**Execution Type Categories:**
- **Native Executables**: Most modules work (process injection, registry, etc.)
- **PDF Documents**: Only PDF-specific modules (pdf-text, pdf-hyperlink, etc.)
- **LNK Shortcuts**: Only LNK-specific modules (lnk-command, lnk-url)
- **PowerShell Scripts**: PowerShell-compatible and general modules
- **Excel Macros**: Excel-specific modules (excel-web-request, etc.)

### Testing Framework

Wildfire Buffet includes a comprehensive test suite with **75 tests** covering all framework functionality:

```bash
# Run all tests
dotnet test

# Or run the test project directly
cd WildfireBuffet.Tests
dotnet run
```

**Test Coverage (100% Passing):**
- ✅ **Module Loading & Discovery** - Validates module parsing and metadata extraction
- ✅ **Execution Type Discovery** - Tests all 6 execution types (native-exe, PDF, LNK, PowerShell, Excel)
- ✅ **Packager Discovery** - Tests ZIP packaging and custom packagers
- ✅ **Parameter Substitution** - Validates all parameter types (string, int, bool, stringlist, choice)
- ✅ **Polymorphic Output** - Ensures unique hashes for each sample
- ✅ **Configuration Handling** - Tests JSON config loading and validation
- ✅ **Build Pipeline** - End-to-end build tests for all execution types
- ✅ **Module Compatibility** - Verifies modules work with their declared execution types

**Test Architecture:**
- Uses xUnit with collection fixtures for shared resources
- Prevents "Assembly already loaded" conflicts via shared TestFixture
- Each test runs in isolated temp directory
- Tests validate both compilation and runtime behavior

Example test output:
```
Starting test execution, please wait...
A total of 1 test files matched the specified pattern.

Passed!  - Failed:     0, Passed:    75, Skipped:     0, Total:    75
```

## Usage Workflow

### 1. Launch the Builder

```bash
cd WildfireBuffet.CLI
dotnet run
```

### 2. Select Modules

The interactive menu lets you:
- Browse available modules
- Select which ones to include
- Configure parameters for each module

### 3. Configure Build

- **Sample Count**: How many unique executables to generate (1-1000)
- **Randomization Level**: 0-10 (higher = more polymorphic)
- **Base Name**: Prefix for output files
- **Output Directory**: Where to save generated files

### 4. Generate!

Watch as unique payloads are created with different hashes:

```
Building 10 payloads...
✓ #1 - test_0001.exe - 33.5 MB - Hash: a3f2b1c4...
✓ #2 - test_0002.exe - 33.5 MB - Hash: d8e9f0a1...
✓ #3 - test_0003.exe - 33.5 MB - Hash: 4b7c2a91...
...
✓ 10 succeeded
```

### 5. Save Configuration (Optional)

After a successful build, the interactive CLI will ask if you want to save the configuration:

```
Save this configuration to a JSON file? [y/n]
Config file path? [build-config.json]
```

This allows you to:
- Reuse the same configuration later
- Automate builds using command-line mode
- Share configurations with team members
- Version control your build recipes

## Polymorphic Techniques

The PolymorphicEngine applies several randomization techniques to ensure unique hashes:

- 🔀 **Variable Name Obfuscation** - Random identifier generation
- 🎲 **Code Block Shuffling** - Randomize execution order
- 📦 **Junk Code Injection** - Meaningless but valid code
- 🔤 **String Encoding** - Multiple encoding methods
- 🏗️ **Random Namespaces/Classes** - Unique type names per sample
- ⏱️ **Random Delays** - Variable timing between operations

### Randomization Levels

- **0-2**: Minimal randomization, basic obfuscation
- **3-5**: Moderate randomization, junk code injection
- **6-8**: High randomization, aggressive shuffling
- **9-10**: Maximum randomization, all techniques enabled

## Advanced Features

### Dynamic Import Resolution

The DynamicImportResolver module hides Win32 API imports from static analysis:

```csharp
/*
 * PARAMETERS:
 * @DllName:string:kernel32.dll:::DLL to load
 * @ApiNames:stringlist:GetSystemInfo,Sleep:::APIs to resolve
 */
```

This generates code that uses `LoadLibrary` and `GetProcAddress` to resolve APIs at runtime, making static analysis more difficult.

### Module Parameter Configuration

When you select modules with parameters, the builder will prompt you:

```
Configure CPU Core Check:
  Minimum CPU cores required? [2]

Configure URL Visitor:
  Comma-separated list of URLs to visit? [https://www.google.com]
```

## Security & Legal Notice

⚠️ **AUTHORIZED USE ONLY** ⚠️

This tool is designed for **authorized security testing only**. Valid use cases:

- Testing security product detection capabilities
- Red team exercises with proper authorization
- Security research and education
- CTF competitions
- Defensive security training

**Do not use for malicious purposes.** Always obtain proper authorization before testing.

## How Hot-Reload Works

Wildfire Buffet's hot-reload architecture means you can add new modules, execution types, and packagers **without recompiling the framework**. Just drop `.cs` files into the appropriate directories and they're automatically discovered at runtime.

### Architecture Overview

```
┌─────────────────────────────────────────────────────────────┐
│                    WILDFIRE BUFFET FRAMEWORK                 │
├─────────────────────────────────────────────────────────────┤
│                                                              │
│  ┌──────────────┐    ┌──────────────┐    ┌──────────────┐  │
│  │   Modules/   │    │ExecutionTypes│    │  Packagers/  │  │
│  │              │    │              │    │              │  │
│  │ - Evasions/  │    │ - *.cs files │    │ - *.cs files │  │
│  │ - Actions/   │    │              │    │              │  │
│  └──────┬───────┘    └──────┬───────┘    └──────┬───────┘  │
│         │                   │                   │           │
│         ▼                   ▼                   ▼           │
│  ┌──────────────────────────────────────────────────────┐  │
│  │           DISCOVERY PROVIDERS (Hot-Reload)            │  │
│  ├──────────────────────────────────────────────────────┤  │
│  │ FileBasedModuleLoader │ ExecutionTypeProvider │       │  │
│  │                       │ PackagerProvider      │       │  │
│  └──────────────────────────────────────────────────────┘  │
│         │                   │                   │           │
│         └─────────┬─────────┴─────────┬─────────┘           │
│                   ▼                   ▼                     │
│         ┌─────────────────────────────────────┐             │
│         │   PayloadBuilder (Orchestration)    │             │
│         └─────────────────────────────────────┘             │
│                           │                                 │
│                           ▼                                 │
│         ┌─────────────────────────────────────┐             │
│         │  Roslyn (Dynamic Compilation)       │             │
│         │  + dotnet publish                   │             │
│         └─────────────────────────────────────┘             │
│                           │                                 │
│                           ▼                                 │
│                   Generated Output                          │
│              (.exe, .pdf, .lnk, .ps1, .xlsm)                │
└─────────────────────────────────────────────────────────────┘
```

### Component 1: Module Discovery (FileBasedModuleLoader)

**Location:** `WildfireBuffet.Core/Discovery/FileBasedModuleLoader.cs`

**How It Works:**
1. **Scans** the `Modules/` directory tree recursively for all `.cs` files
2. **Parses** each file using `ModuleFileParser` to extract:
   - Module metadata (name, ID, description, category)
   - Parameter definitions with types and validation rules
   - Using statements (for dependency resolution)
   - Code blocks (the actual C# code to inject)
3. **Creates** `ModuleDefinition` objects for each valid module
4. **Stores** in `AllModules` collection, available immediately

**Key Feature:** Modules are plain `.cs` files with special comment headers - no compilation needed at this stage!

**Example Module Discovery:**
```csharp
// FileBasedModuleLoader.LoadModules("Modules/")
// 1. Finds: Modules/Evasions/CpuCoreCheck.cs
// 2. Reads file contents
// 3. ModuleFileParser.Parse() extracts:
//    - ID: "cpu-core-check"
//    - Name: "CPU Core Check"
//    - Parameters: @MinCores:int:2
//    - Code: if (Environment.ProcessorCount < @MinCores) ...
// 4. Creates ModuleDefinition with all metadata
// 5. Available immediately in UI dropdown!
```

### Component 2: Execution Type Discovery (ExecutionTypeProvider)

**Location:** `WildfireBuffet.Core/Discovery/ExecutionTypeProvider.cs`

**How It Works:**
1. **Scans** the `ExecutionTypes/` directory for `.cs` files
2. **Compiles** each file **dynamically** using Roslyn (`Microsoft.CodeAnalysis.CSharp`)
3. **Loads** assemblies into isolated `AssemblyLoadContext`
4. **Reflects** to find classes implementing `IExecutionType` interface
5. **Instantiates** and stores execution type implementations
6. **Reads** descriptor metadata (ID, name, supported modules, file extension)

**Key Feature:** Execution types are compiled at runtime, so you can add new output formats without rebuilding the framework!

**Example Execution Type Loading:**
```csharp
// ExecutionTypeProvider.LoadExecutionTypes()
// 1. Finds: ExecutionTypes/PdfDocument.cs
// 2. Uses Roslyn to compile in-memory:
//    - CSharpCompilation.Create()
//    - Adds references to required assemblies (QuestPDF, PdfSharp, etc.)
//    - Emits to MemoryStream
// 3. AssemblyLoadContext.LoadFromStream() loads compiled assembly
// 4. Reflection finds class PdfDocument : ExecutionTypeBase
// 5. Reads Descriptor property:
//    - Id: "pdf-document"
//    - FileExtension: ".pdf"
//    - SupportedModules: ["pdf-text", "pdf-hyperlink", "pdf-javascript", "pdf-image"]
// 6. Available immediately in UI!
```

**Important:** Because compilation happens at runtime, you can:
- Add new execution types without rebuilding the framework
- Use any NuGet packages (they're included via PackageReference in the main project)
- Test immediately after dropping the file

### Component 3: Packager Discovery (PackagerProvider)

**Location:** `WildfireBuffet.Core/Discovery/PackagerProvider.cs`

**How It Works:**
1. **Scans** the `Packagers/` directory for `.cs` files
2. **Compiles** dynamically using Roslyn (same as execution types)
3. **Loads** via `AssemblyLoadContext`
4. **Finds** classes implementing `IPackager` interface
5. **Stores** packager implementations for post-processing

**Key Feature:** Packagers process the generated output (e.g., ZIP compression, padding, encryption).

**Example Packager Loading:**
```csharp
// PackagerProvider.LoadPackagers()
// 1. Finds: Packagers/ZipArchivePackager.cs
// 2. Compiles with Roslyn
// 3. Loads ZipArchivePackager : PackagerBase
// 4. Reads Descriptor:
//    - Id: "zip-archive"
//    - Name: "ZIP Archive"
//    - FileExtension: ".zip"
// 5. Available for post-build packaging!
```

### Component 4: Parameter Substitution System

**Location:** `WildfireBuffet.Core/Models/ModuleDefinition.cs`

**How It Works:**
1. Module code contains **placeholders** like `@MinCores`, `@Urls`, `@Operation`
2. User configures values via UI or JSON config
3. `GetSubstitutedCode()` method replaces placeholders with actual values:
   - Strings → Quoted: `"https://example.com"`
   - Integers → Raw: `8`
   - Booleans → Lowercase: `true` or `false`
   - StringLists → C# array: `new[] { "url1.com", "url2.com" }`
   - Choice → Selected value: `"Read"`
4. Result is valid C# code ready for compilation

**Example:**
```csharp
// Original module code:
if (Environment.ProcessorCount < @MinCores) { ... }

// User sets MinCores = 8
// GetSubstitutedCode() produces:
if (Environment.ProcessorCount < 8) { ... }
```

### Component 5: Build Pipeline (PayloadBuilder)

**Location:** `WildfireBuffet.Core/Builder/PayloadBuilder.cs`

**The Build Process:**

```
1. USER CONFIGURATION
   ├─ Select execution type (e.g., "native-executable")
   ├─ Select modules (e.g., cpu-core-check, url-visitor)
   ├─ Configure parameters (MinCores: 8, Urls: "example.com")
   └─ Set randomization level (0-10)

2. CODE GENERATION
   ├─ Get execution type implementation
   ├─ For each module:
   │  ├─ Substitute parameters (@MinCores → 8)
   │  ├─ Extract using statements
   │  └─ Extract code blocks
   ├─ Call execution type's Generate() method
   │  ├─ Builds complete C# file with:
   │  │  ├─ Using statements
   │  │  ├─ Namespace/class structure
   │  │  ├─ Module code injected in order
   │  │  └─ Polymorphic variations (per sample)
   │  └─ Returns byte[] (for PDFs) or source code (for native EXE)

3. POLYMORPHIC RANDOMIZATION (per sample)
   ├─ Variable name obfuscation (var123, temp456)
   ├─ Random comments and whitespace
   ├─ Code block shuffling (where safe)
   ├─ Junk code injection (dead code)
   └─ Random delays and sleep times

4. COMPILATION (for native executables)
   ├─ Create temporary directory
   ├─ Write generated .cs file
   ├─ Create .csproj file with:
   │  ├─ RuntimeIdentifier: win-x64
   │  ├─ PublishSingleFile: true
   │  ├─ SelfContained: true
   │  └─ Required NuGet packages
   ├─ Execute: dotnet publish
   └─ Collect output .exe file

5. PACKAGING (optional)
   ├─ If packager specified:
   │  ├─ Call packager's Package() method
   │  ├─ Apply transformations (ZIP, padding, etc.)
   │  └─ Return packaged output
   └─ Compute SHA256 hash

6. OUTPUT
   ├─ Save to configured output directory
   ├─ Filename: baseName_0001.{extension}
   └─ Record metadata (hash, size, sample number)
```

### Why No Framework Recompilation Needed?

**Traditional Approach (compile-time):**
```
Add Module → Edit Source Code → Recompile Entire Framework → Restart App → Test
   ❌ Slow    ❌ Error-prone    ❌ Heavyweight            ❌ Friction
```

**Wildfire Buffet Approach (hot-reload):**
```
Add Module → Drop .cs File → Refresh/Restart → Available Immediately
   ✅ Fast    ✅ Simple        ✅ No Compilation    ✅ Developer-Friendly
```

**Technical Implementation:**

1. **Modules** - Not compiled, just parsed as text
   - Metadata extracted from comments
   - Code stored as strings
   - Substituted at build time

2. **Execution Types** - Compiled at runtime with Roslyn
   - `CSharpCompilation.Create()` compiles C# source to assembly
   - `AssemblyLoadContext` loads without affecting main app domain
   - Reflection discovers `IExecutionType` implementations
   - Interface contract ensures compatibility

3. **Packagers** - Same as execution types
   - Compiled dynamically
   - Loaded via `AssemblyLoadContext`
   - Implements `IPackager` interface

**Key Technologies:**
- **Roslyn (Microsoft.CodeAnalysis.CSharp)** - Compile C# code at runtime
- **AssemblyLoadContext** - Isolated assembly loading without conflicts
- **Reflection** - Discover types and interfaces dynamically
- **dotnet CLI** - Publish single-file executables

### Hot-Reload Workflow Example

**Adding a New Execution Type:**

1. Create `ExecutionTypes/MyNewFormat.cs`:
```csharp
using WildfireBuffet.Core.Abstractions;

public class MyNewFormat : ExecutionTypeBase
{
    public override ExecutionTypeDescriptor Descriptor => new()
    {
        Id = "my-new-format",
        Name = "My New Format",
        FileExtension = ".custom",
        SupportedModules = new[] { "*" }
    };

    public override async Task<byte[]> GenerateAsync(ExecutionContext context)
    {
        // Generate output in your custom format
        return await Task.FromResult(customBytes);
    }
}
```

2. Restart the app (or refresh in future versions)
3. **Execution type immediately appears in UI!**
4. No framework recompilation needed

**The Magic:**
- `ExecutionTypeProvider` scans `ExecutionTypes/` on startup
- Compiles `MyNewFormat.cs` with Roslyn
- Loads into isolated `AssemblyLoadContext`
- Discovers via reflection: `typeof(IExecutionType).IsAssignableFrom(type)`
- Instantiates and adds to `ExecutionTypes` collection
- UI reads from `ExecutionTypes` → new option appears!

## Technical Architecture

### Module Discovery

1. **Scan** - FileBasedModuleLoader scans `Modules/` directories
2. **Parse** - ModuleFileParser extracts metadata and code
3. **Load** - Modules become available in the UI

### Build Pipeline

1. **Module Selection** - User picks modules via interactive UI or config file
2. **Parameter Configuration** - Configure each module's parameters
3. **Code Generation** - Inject module code with parameter substitution
4. **Polymorphism** - Apply randomization techniques
5. **Compilation** - Create temporary .csproj and use `dotnet publish`
6. **Packaging** - Generate self-contained single-file executable
7. **Output** - Save with unique hash

### Compilation Details

Generated payloads are created using `dotnet publish` with these settings:
- **Self-Contained**: Includes .NET runtime (no installation required)
- **Single-File**: All dependencies bundled into one executable
- **RuntimeIdentifier**: win-x64 (Windows 64-bit)
- **Compressed**: Single-file compression enabled
- **Typical Size**: 30-35 MB per executable

This ensures payloads run on any Windows x64 machine without dependencies.

### How Modules Work

```
┌─────────────────┐
│  MyCheck.cs     │
│  (in Modules/)  │
└────────┬────────┘
         │
         ▼
┌─────────────────┐
│ ModuleFileParser│ ◄── Extracts metadata
│                 │ ◄── Extracts parameters
│                 │ ◄── Extracts code
└────────┬────────┘
         │
         ▼
┌─────────────────┐
│ ModuleDefinition│
│ • Name          │
│ • Parameters    │
│ • Code          │
└────────┬────────┘
         │
         ▼
┌─────────────────┐
│ PayloadBuilder  │ ◄── Substitutes @Parameters
│                 │ ◄── Applies polymorphism
│                 │ ◄── Compiles with Roslyn
└────────┬────────┘
         │
         ▼
    payload.exe
```

## Troubleshooting

### No modules found

Make sure the `Modules/` directory exists and contains `.cs` files:
```bash
ls Modules/Evasions/
ls Modules/Actions/
```

### Module parsing errors

Check that your module file follows the format:
- Header comment with MODULE, ID, DESCRIPTION, CATEGORY
- Parameters in format: `@Name:type:default:min:max:description`
- `// USINGS` section
- `// CODE` section

### Compilation errors

If generated payloads fail to compile:
- Check that all `@Parameters` in CODE section are defined in PARAMETERS
- Ensure USINGS section includes all required namespaces
- Test your code manually first

## Extending the Framework

### Add New Parameter Types

Edit `WildfireBuffet.Core/Models/ModuleParameterDefinition.cs`:

```csharp
public enum ParameterType
{
    String,
    Int,
    Bool,
    StringList,
    Float,     // Add new type
    FileList   // Add new type
}
```

### Custom Polymorphic Techniques

Edit `WildfireBuffet.Core/Engine/PolymorphicEngine.cs` to add new randomization methods.

### Output Modules

Output modules work differently - they process the compiled executable:
- ZIP packaging
- ISO creation
- Code signing
- PE header modification

(Coming soon - output modules currently under development)

## Future Enhancements

**Already Implemented:**
- ✅ Multiple execution types (native-exe, PDF, LNK, PowerShell, Excel)
- ✅ Comprehensive test framework (75 tests, 100% passing)
- ✅ Cross-platform UI (Avalonia 11)
- ✅ Packaging system (ZIP, padding)
- ✅ Choice parameters (dropdowns)
- ✅ Hot-reload for execution types and packagers

**Planned:**
- [ ] More evasion techniques (mouse movement, user interaction)
- [ ] PE header manipulation
- [ ] Code signing support
- [ ] Multi-threaded generation
- [ ] REST API for automation
- [ ] File format validation and linting
- [ ] Real-time module refresh (no restart needed)

## Contributing

Contributions welcome! To add modules:

1. Create your `.cs` module file
2. Follow the format guidelines
3. Test thoroughly
4. Share your modules!

##License

For educational and authorized security testing purposes only.

---

**Remember:** With great power comes great responsibility. Use ethically. 🛡️
