/*
 * PACKAGER: ZIP Archive
 * ID: zip-archive
 * DESCRIPTION: Wraps files into ZIP archives with optional size padding
 * CATEGORY: Archive
 * FILE EXTENSION: .zip
 *
 * PARAMETERS:
 * @TargetSizeMB:int:0:::Target size in MB (0 = no padding, adds junk files to reach size)
 * @Password:string::::Optional ZIP password protection
 * @CompressionLevel:string:Optimal:::Compression level: NoCompression, Fastest, Optimal, SmallestSize
 *
 * SUPPORTED EXECUTION TYPES: * (all)
 *
 * HOW TO USE:
 * This packager wraps any execution type output into a ZIP archive.
 * - Set TargetSizeMB to inflate file size (evades size-based detection)
 * - Use Password for password-protected archives
 */

using System;
using System.IO;
using System.IO.Compression;
using System.Text;
using System.Threading.Tasks;
using System.Collections.Generic;
using WildfireBuffet.Core.Abstractions;

public class ZipPackager : PackagerBase
{
    public override PackagerDescriptor Descriptor => new()
    {
        Id = "zip-archive",
        Name = "ZIP Archive",
        Description = "Wraps files into ZIP archives with optional size padding",
        FileExtension = ".zip",
        Icon = "📦",
        Category = "Archive",
        SupportedExecutionTypes = new[] { "*" }, // Supports all execution types
        ConfigurableParameters = new Dictionary<string, string>
        {
            { "TargetSizeMB", "int - Target size in MB (0 = no padding)" },
            { "Password", "string - Optional password protection (currently not supported)" },
            { "CompressionLevel", "string - NoCompression, Fastest, Optimal, SmallestSize" }
        }
    };

    public override async Task<PackagingResult> PackageAsync(PackagingContext context)
    {
        // Validate execution type compatibility
        var validationError = ValidateExecutionTypeCompatibility(context);
        if (validationError != null)
        {
            return validationError;
        }

        // Get configuration
        var targetSizeMB = GetConfig(context, "TargetSizeMB", 0);
        var compressionLevelStr = GetConfig(context, "CompressionLevel", "Optimal");

        // Parse compression level
        if (!Enum.TryParse<CompressionLevel>(compressionLevelStr, true, out var compressionLevel))
        {
            compressionLevel = CompressionLevel.Optimal;
        }

        try
        {
            using var memoryStream = new MemoryStream();

            // Create ZIP archive
            using (var archive = new ZipArchive(memoryStream, ZipArchiveMode.Create, leaveOpen: true))
            {
                // Add the main payload file
                var mainFileName = $"{context.BaseName}{context.InputExtension}";
                var entry = archive.CreateEntry(mainFileName, compressionLevel);

                using (var entryStream = entry.Open())
                {
                    await entryStream.WriteAsync(context.InputFile, 0, context.InputFile.Length);
                }

                // Add padding files if target size specified
                if (targetSizeMB > 0)
                {
                    var targetBytes = (long)targetSizeMB * 1024 * 1024;
                    var currentSize = memoryStream.Length;

                    if (currentSize < targetBytes)
                    {
                        AddPaddingFiles(archive, targetBytes - currentSize, compressionLevel);
                    }
                }
            }

            var result = PackagingResult.Succeeded(memoryStream.ToArray());
            result.Diagnostics.Add($"Created ZIP archive: {memoryStream.Length:N0} bytes");

            if (targetSizeMB > 0)
            {
                result.Diagnostics.Add($"Target size: {targetSizeMB} MB, Actual: {memoryStream.Length / 1024.0 / 1024.0:F2} MB");
            }

            return result;
        }
        catch (Exception ex)
        {
            return PackagingResult.Failed($"Failed to create ZIP archive: {ex.Message}");
        }
    }

    /// <summary>
    /// Add padding files to inflate archive size
    /// </summary>
    private void AddPaddingFiles(ZipArchive archive, long paddingSize, CompressionLevel level)
    {
        const long maxChunkSize = 50 * 1024 * 1024; // 50MB chunks
        var random = new Random(42);
        var remainingSize = paddingSize;
        var fileIndex = 0;

        while (remainingSize > 0)
        {
            var chunkSize = Math.Min(remainingSize, maxChunkSize);
            var paddingFileName = $".cache/data_{fileIndex:D4}.bin";

            var entry = archive.CreateEntry(paddingFileName, CompressionLevel.NoCompression); // Don't compress padding
            using (var entryStream = entry.Open())
            {
                // Write padding in 1MB chunks to avoid memory issues
                const int bufferSize = 1024 * 1024;
                var buffer = new byte[bufferSize];
                var written = 0L;

                while (written < chunkSize)
                {
                    var toWrite = (int)Math.Min(bufferSize, chunkSize - written);
                    random.NextBytes(buffer);
                    entryStream.Write(buffer, 0, toWrite);
                    written += toWrite;
                }
            }

            remainingSize -= chunkSize;
            fileIndex++;
        }
    }
}
