/*
 * PACKAGER: ISO Image
 * ID: iso-image
 * DESCRIPTION: Creates ISO 9660 disc images with optional size padding
 * CATEGORY: Archive
 * FILE EXTENSION: .iso
 *
 * PARAMETERS:
 * @VolumeLabel:string:PAYLOAD_DISC:::Volume label for the ISO filesystem
 * @TargetSizeMB:int:0:::Target size in MB (0 = no padding, adds junk files to reach size)
 * @FileSystem:string:ISO9660:::Filesystem type: ISO9660 (standard)
 *
 * SUPPORTED EXECUTION TYPES: * (all)
 *
 * HOW TO USE:
 * This packager wraps any execution type output into an ISO disc image.
 * - Set VolumeLabel to customize the disc name
 * - Set TargetSizeMB to inflate file size (evades size-based detection)
 */

using System;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using System.Collections.Generic;
using WildfireBuffet.Core.Abstractions;
using DiscUtils;
using DiscUtils.Iso9660;

public class IsoPackager : PackagerBase
{
    public override PackagerDescriptor Descriptor => new()
    {
        Id = "iso-image",
        Name = "ISO Image",
        Description = "Creates ISO 9660 disc images with optional size padding",
        FileExtension = ".iso",
        Icon = "💿",
        Category = "Archive",
        SupportedExecutionTypes = new[] { "*" }, // Supports all execution types
        ConfigurableParameters = new Dictionary<string, string>
        {
            { "VolumeLabel", "string - Volume label for the ISO filesystem" },
            { "TargetSizeMB", "int - Target size in MB (0 = no padding)" },
            { "FileSystem", "string - Filesystem type (ISO9660 only currently)" }
        }
    };

    public override async Task<PackagingResult> PackageAsync(PackagingContext context)
    {
        // Validate execution type compatibility
        var validationError = ValidateExecutionTypeCompatibility(context);
        if (validationError != null)
        {
            return validationError;
        }

        // Get configuration
        var volumeLabel = GetConfig(context, "VolumeLabel", "PAYLOAD_DISC");
        var targetSizeMB = GetConfig(context, "TargetSizeMB", 0);

        // Sanitize volume label (ISO9660 max 32 chars, uppercase alphanumeric + underscore)
        volumeLabel = SanitizeVolumeLabel(volumeLabel);

        try
        {
            using var memoryStream = new MemoryStream();

            // Create ISO builder
            var builder = new CDBuilder
            {
                UseJoliet = true, // Enable Joliet for long filenames
                VolumeIdentifier = volumeLabel
            };

            // Add the main payload file
            var mainFileName = $"{context.BaseName}{context.InputExtension}";
            builder.AddFile(mainFileName, context.InputFile);

            // Add padding files if target size specified
            if (targetSizeMB > 0)
            {
                var targetBytes = (long)targetSizeMB * 1024 * 1024;
                AddPaddingFiles(builder, targetBytes, context.InputFile.Length);
            }

            // Build the ISO image
            builder.Build(memoryStream);

            // Seek to beginning
            memoryStream.Position = 0;

            var result = PackagingResult.Succeeded(memoryStream.ToArray());
            result.Diagnostics.Add($"Created ISO image: {memoryStream.Length:N0} bytes");
            result.Diagnostics.Add($"Volume Label: {volumeLabel}");

            if (targetSizeMB > 0)
            {
                result.Diagnostics.Add($"Target size: {targetSizeMB} MB, Actual: {memoryStream.Length / 1024.0 / 1024.0:F2} MB");
            }

            return result;
        }
        catch (Exception ex)
        {
            return PackagingResult.Failed($"Failed to create ISO image: {ex.Message}");
        }
    }

    /// <summary>
    /// Sanitize volume label for ISO9660 compliance
    /// </summary>
    private string SanitizeVolumeLabel(string label)
    {
        if (string.IsNullOrWhiteSpace(label))
        {
            label = "PAYLOAD_DISC";
        }

        // Convert to uppercase and replace invalid characters
        var sb = new StringBuilder();
        foreach (var c in label.ToUpperInvariant())
        {
            if (char.IsLetterOrDigit(c) || c == '_')
            {
                sb.Append(c);
            }
            else
            {
                sb.Append('_');
            }
        }

        // Truncate to 32 characters (ISO9660 limit)
        var sanitized = sb.ToString();
        if (sanitized.Length > 32)
        {
            sanitized = sanitized.Substring(0, 32);
        }

        return sanitized;
    }

    /// <summary>
    /// Add padding files to inflate ISO size
    /// </summary>
    private void AddPaddingFiles(CDBuilder builder, long targetBytes, long currentPayloadSize)
    {
        // Account for existing payload (no decoy files)
        const long isoOverhead = 64 * 1024; // ~64KB for ISO filesystem overhead

        var currentEstimatedSize = currentPayloadSize + isoOverhead;
        var paddingNeeded = targetBytes - currentEstimatedSize;

        if (paddingNeeded <= 0)
        {
            return; // Already at or above target size
        }

        const long maxChunkSize = 50 * 1024 * 1024; // 50MB chunks
        var random = new Random(42); // Deterministic padding
        var remainingSize = paddingNeeded;
        var fileIndex = 0;

        while (remainingSize > 0)
        {
            var chunkSize = Math.Min(remainingSize, maxChunkSize);
            var paddingFileName = $"DATA/CACHE_{fileIndex:D4}.BIN";

            // Generate padding bytes
            var paddingData = new byte[chunkSize];
            random.NextBytes(paddingData);

            builder.AddFile(paddingFileName, paddingData);

            remainingSize -= chunkSize;
            fileIndex++;
        }
    }
}
