/*
 * MODULE: Mutex Checker
 * ID: mutex-checker
 * DESCRIPTION: Creates or checks for a named mutex - common malware technique for single-instance enforcement
 * CATEGORY: Process Synchronization
 * SUPPORTED_EXECUTION_TYPES: native-executable,native-executable-aot,powershell-script
 *
 * PARAMETERS:
 * @MutexName|string|Global\AppMutex|||required|Name of the mutex (can include Global\ or Local\ prefix)
 * @ExitIfExists|bool|true|||Exit if mutex already exists (prevents multiple instances)
 * @HoldTimeSeconds|int|5|1|300||Time to hold the mutex in seconds
 * @ShowStatus|bool|true|||Show detailed status messages
 */

// USINGS
using System;
using System.Threading;

// CODE
try
{
    var mutexName = @MutexName;
    var exitIfExists = @ExitIfExists;
    var holdTimeSeconds = @HoldTimeSeconds;
    var showStatus = @ShowStatus;

    if (showStatus)
    {
        Console.WriteLine("[*] Mutex Checker - Starting...");
        Console.WriteLine($"[*] Mutex Name: {mutexName}");
        Console.WriteLine($"[*] Exit If Exists: {exitIfExists}");
        Console.WriteLine($"[*] Hold Time: {holdTimeSeconds} seconds");
        Console.WriteLine();
    }

    // Try to create or open the mutex
    bool createdNew;
    Mutex mutex = null;

    try
    {
        mutex = new Mutex(true, mutexName, out createdNew);

        if (createdNew)
        {
            if (showStatus)
            {
                Console.WriteLine($"[+] Successfully created NEW mutex: {mutexName}");
                Console.WriteLine("[*] This is the FIRST instance of this application");
            }

            // Hold the mutex for the specified time
            if (showStatus)
            {
                Console.WriteLine($"[*] Holding mutex for {holdTimeSeconds} seconds...");
            }

            Thread.Sleep(TimeSpan.FromSeconds(holdTimeSeconds));

            if (showStatus)
            {
                Console.WriteLine("[+] Mutex hold time completed");
            }

            // Release and dispose the mutex
            mutex.ReleaseMutex();
            mutex.Dispose();

            if (showStatus)
            {
                Console.WriteLine($"[+] Mutex released and disposed");
            }
        }
        else
        {
            if (showStatus)
            {
                Console.WriteLine($"[!] Mutex ALREADY EXISTS: {mutexName}");
                Console.WriteLine("[!] Another instance is already running!");
            }

            if (exitIfExists)
            {
                if (showStatus)
                {
                    Console.WriteLine("[*] ExitIfExists=true - Terminating this instance");
                }

                // Clean up
                if (mutex != null)
                {
                    mutex.Dispose();
                }

                // Exit immediately to prevent multiple instances
                Environment.Exit(1);
            }
            else
            {
                if (showStatus)
                {
                    Console.WriteLine("[*] ExitIfExists=false - Continuing anyway");
                    Console.WriteLine($"[*] Holding for {holdTimeSeconds} seconds...");
                }

                Thread.Sleep(TimeSpan.FromSeconds(holdTimeSeconds));

                if (mutex != null)
                {
                    mutex.Dispose();
                }

                if (showStatus)
                {
                    Console.WriteLine("[*] Done");
                }
            }
        }
    }
    catch (UnauthorizedAccessException)
    {
        Console.WriteLine($"[-] Access denied to mutex: {mutexName}");
        Console.WriteLine("[-] May require different privileges or namespace");
    }
    catch (Exception ex)
    {
        Console.WriteLine($"[-] Error with mutex: {ex.Message}");
    }

    if (showStatus)
    {
        Console.WriteLine("[+] Mutex checker completed");
    }
}
catch (Exception ex)
{
    Console.WriteLine($"[-] Unexpected error: {ex.Message}");
    Console.WriteLine($"[-] Stack trace: {ex.StackTrace}");
}
