/*
 * EXECUTION TYPE: PowerShell Script
 * ID: powershell-script
 * DESCRIPTION: PowerShell .ps1 script that executes module code with polymorphic output
 * CATEGORY: Script
 * FILE EXTENSION: .ps1
 *
 * PARAMETERS:
 * @ExecutionPolicy:choice:Bypass:Bypass|RemoteSigned|Unrestricted:::Execution policy for the script
 * @RequireAdmin:bool:false:::Require administrator privileges
 * @NoProfile:bool:true:::Run without loading PowerShell profile
 * @NonInteractive:bool:false:::Run in non-interactive mode
 * @AddRandomComments:bool:true:::Add random comments for polymorphism
 *
 * SUPPORTED MODULES: * (all modules marked with powershell-script execution type)
 *
 * HOW IT WORKS:
 * Converts C# module logic to PowerShell equivalents with advanced polymorphism:
 * - Randomizes variable names, function names, comment styles
 * - Varies whitespace, string quote styles, syntax variations
 * - Each generated .ps1 has different hash even with same recipe
 * - Supports all standard PowerShell features and .NET interop
 *
 * POLYMORPHIC FEATURES:
 * - Random variable naming schemes (camelCase, PascalCase, snake_case)
 * - Random comment insertion (single-line, block, inline)
 * - Varied PowerShell syntax (Write-Host vs Write-Output, aliases)
 * - Random whitespace and indentation variations
 * - Different quote styles (single vs double quotes)
 * - Random function and parameter ordering
 */

using System;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Collections.Generic;
using WildfireBuffet.Core.Abstractions;
using WildfireBuffet.Core.Models;
using WildfireBuffet.Core.Helpers;

public class PowerShellScript : ExecutionTypeBase
{
    // Polymorphic variable name styles
    private static readonly string[][] _variableStyles = new[]
    {
        new[] { "var", "val", "item", "obj", "data", "result" },
        new[] { "x", "y", "z", "a", "b", "c" },
        new[] { "temp", "tmp", "buffer", "cache", "holder" },
        new[] { "element", "entity", "record", "entry", "node" }
    };

    // Random comment styles
    private static readonly string[] _commentPrefixes = new[]
    {
        "# ",
        "## ",
        "#-- ",
        "#=> ",
        "#:: "
    };

    // Random PowerShell output cmdlets
    private static readonly string[] _outputCmdlets = new[]
    {
        "Write-Host",
        "Write-Output",
        "echo"
    };

    public override ExecutionTypeDescriptor Descriptor => new()
    {
        Id = "powershell-script",
        Name = "PowerShell Script",
        Description = "PowerShell .ps1 script with polymorphic output and module code execution",
        FileExtension = ".ps1",
        Icon = "⚡",
        Category = "Script",
        SupportedModules = new[] { "*" },
        ConfigurableParameters = new Dictionary<string, string>
        {
            { "ExecutionPolicy", "choice - Bypass|RemoteSigned|Unrestricted" },
            { "RequireAdmin", "bool - Require administrator privileges" },
            { "NoProfile", "bool - Run without loading profile" },
            { "NonInteractive", "bool - Run in non-interactive mode" },
            { "AddRandomComments", "bool - Add random comments for polymorphism" }
        }
    };

    public override async Task<ExecutionResult> GenerateAsync(ExecutionContext context)
    {
        // Validate module compatibility
        var validationError = ValidateModuleCompatibility(context);
        if (validationError != null)
        {
            return validationError;
        }

        // Get configuration
        var executionPolicy = GetConfig(context, "ExecutionPolicy", "Bypass");
        var requireAdmin = GetConfig(context, "RequireAdmin", false);
        var noProfile = GetConfig(context, "NoProfile", true);
        var nonInteractive = GetConfig(context, "NonInteractive", false);
        var addRandomComments = GetConfig(context, "AddRandomComments", true);

        try
        {
            // Create deterministic random for reproducible variation
            var random = CreatePolymorphicRandom(context.SampleNumber);

            // Generate PowerShell script with polymorphism
            var scriptContent = GeneratePowerShellScript(
                context,
                executionPolicy,
                requireAdmin,
                noProfile,
                nonInteractive,
                addRandomComments,
                random
            );

            // Convert to UTF-8 bytes with BOM
            var scriptBytes = Encoding.UTF8.GetBytes(scriptContent);

            var result = ExecutionResult.Succeeded(scriptBytes);
            result.Diagnostics.Add($"Generated PowerShell script with {context.Modules.Count} module(s)");
            result.Diagnostics.Add($"Execution Policy: {executionPolicy}");
            result.Diagnostics.Add($"Require Admin: {requireAdmin}");
            result.Diagnostics.Add($"Polymorphic seed: {context.SampleNumber}");
            if (addRandomComments)
            {
                result.Diagnostics.Add("Polymorphic features: random comments, variable names, syntax variations");
            }

            return await Task.FromResult(result);
        }
        catch (Exception ex)
        {
            return ExecutionResult.Failed($"PowerShell script generation failed: {ex.Message}\n{ex.StackTrace}");
        }
    }

    /// <summary>
    /// Generate complete PowerShell script with polymorphic variations
    /// </summary>
    private string GeneratePowerShellScript(
        ExecutionContext context,
        string executionPolicy,
        bool requireAdmin,
        bool noProfile,
        bool nonInteractive,
        bool addRandomComments,
        Random random)
    {
        var ps = new StringBuilder();

        // Execution policy bypass header (polymorphic variations)
        if (executionPolicy == "Bypass")
        {
            if (random.Next(2) == 0)
            {
                ps.AppendLine("<# Execution Policy: Bypass #>");
            }
            else
            {
                AddRandomComment(ps, "Set execution policy to bypass restrictions", addRandomComments, random);
            }
        }

        // Script header with polymorphic comments
        AddRandomComment(ps, "Generated by Wildfire Buffet", addRandomComments, random);
        AddRandomComment(ps, $"Sample: {context.SampleNumber}", addRandomComments, random);
        AddRandomComment(ps, $"Timestamp: {DateTime.UtcNow:yyyy-MM-dd HH:mm:ss} UTC", addRandomComments, random);
        ps.AppendLine();

        // Admin requirement check (polymorphic)
        if (requireAdmin)
        {
            GenerateAdminCheck(ps, addRandomComments, random);
        }

        // Error action preference (polymorphic location)
        if (random.Next(2) == 0)
        {
            ps.AppendLine($"$ErrorActionPreference = '{(random.Next(2) == 0 ? "Stop" : "Continue")}'");
            ps.AppendLine();
        }

        // Main execution wrapper (polymorphic try-catch style)
        if (random.Next(2) == 0)
        {
            ps.AppendLine("try {");
        }
        else
        {
            ps.AppendLine("try");
            ps.AppendLine("{");
        }

        AddRandomWhitespace(ps, random);

        // Generate module execution code
        foreach (var module in context.Modules)
        {
            AddRandomComment(ps, $"Module: {module.Name}", addRandomComments, random);
            AddRandomWhitespace(ps, random);

            // Convert module code to PowerShell
            var psModuleCode = ConvertModuleToPowerShell(module, context, addRandomComments, random);
            ps.AppendLine(psModuleCode);

            AddRandomWhitespace(ps, random);
        }

        // Close try block and add catch (polymorphic error handling)
        ps.AppendLine("}");

        if (random.Next(2) == 0)
        {
            ps.AppendLine("catch {");
            ps.AppendLine($"    {GetRandomOutputCmdlet(random)} \"[ERROR] $($_.Exception.Message)\"");
            ps.AppendLine("    exit 1");
            ps.AppendLine("}");
        }
        else
        {
            ps.AppendLine("catch");
            ps.AppendLine("{");
            ps.AppendLine($"    {GetRandomOutputCmdlet(random)} \"[ERROR] Script failed: $($_.Exception.Message)\"");
            ps.AppendLine($"    {GetRandomOutputCmdlet(random)} \"[ERROR] Line: $($_.InvocationInfo.ScriptLineNumber)\"");
            ps.AppendLine("    exit 1");
            ps.AppendLine("}");
        }

        // Success message (polymorphic)
        ps.AppendLine();
        if (random.Next(2) == 0)
        {
            ps.AppendLine($"{GetRandomOutputCmdlet(random)} {GetRandomQuote("Execution completed successfully", random)}");
        }
        else
        {
            ps.AppendLine($"{GetRandomOutputCmdlet(random)} \"\"");
            ps.AppendLine($"{GetRandomOutputCmdlet(random)} {GetRandomQuote("=== Script execution completed ===", random)}");
        }

        // Interactive pause (polymorphic)
        if (!nonInteractive)
        {
            if (random.Next(2) == 0)
            {
                ps.AppendLine($"{GetRandomOutputCmdlet(random)} {GetRandomQuote("Press any key to exit...", random)}");
                ps.AppendLine("$null = $Host.UI.RawUI.ReadKey('NoEcho,IncludeKeyDown')");
            }
            else
            {
                ps.AppendLine("Read-Host -Prompt \"Press Enter to exit\"");
            }
        }

        return ps.ToString();
    }

    /// <summary>
    /// Convert C# module code to PowerShell equivalent
    /// </summary>
    private string ConvertModuleToPowerShell(
        ModuleDefinition module,
        ExecutionContext context,
        bool addRandomComments,
        Random random)
    {
        var ps = new StringBuilder();

        // Module-specific PowerShell conversion based on module ID
        switch (module.Id)
        {
            case "ps-web-request":
                return GeneratePsWebRequestCode(module, addRandomComments, random);

            case "ps-registry-manipulation":
                return GeneratePsRegistryCode(module, addRandomComments, random);

            case "url-visitor":
                // Convert generic URL visitor to PowerShell Invoke-WebRequest
                return GenerateUrlVisitorCode(module, addRandomComments, random);

            default:
                // Generic conversion: attempt to translate C# to PowerShell
                AddRandomComment(ps, $"Generic module: {module.Id}", addRandomComments, random);
                ps.AppendLine($"{GetRandomOutputCmdlet(random)} {GetRandomQuote($"Executing module: {module.Name}", random)}");
                return ps.ToString();
        }
    }

    /// <summary>
    /// Generate PowerShell code for web request module
    /// </summary>
    private string GeneratePsWebRequestCode(ModuleDefinition module, bool addRandomComments, Random random)
    {
        var ps = new StringBuilder();

        // Extract parameters with polymorphic variable names
        var urlsParam = module.ConfiguredValues.GetValueOrDefault("Urls", "https://www.google.com");
        var methodParam = module.ConfiguredValues.GetValueOrDefault("Method", "GET");
        var userAgentParam = module.ConfiguredValues.GetValueOrDefault("UserAgent", "Mozilla/5.0");
        var followRedirectsParam = module.ConfiguredValues.GetValueOrDefault("FollowRedirects", true);

        var urls = urlsParam?.ToString()?.Split(',') ?? new[] { "https://www.google.com" };
        var method = methodParam?.ToString() ?? "GET";
        var userAgent = userAgentParam?.ToString() ?? "Mozilla/5.0";
        var followRedirects = Convert.ToBoolean(followRedirectsParam);

        // Polymorphic variable names
        var varStyle = random.Next(_variableStyles.Length);
        var urlVar = $"${_variableStyles[varStyle][random.Next(_variableStyles[varStyle].Length)]}{random.Next(1, 100)}";
        var responseVar = $"${_variableStyles[varStyle][random.Next(_variableStyles[varStyle].Length)]}{random.Next(100, 200)}";

        AddRandomComment(ps, "Web request execution", addRandomComments, random);

        // Create array of URLs (polymorphic)
        var arrayVar = GetRandomVariableName(random);
        ps.AppendLine($"${arrayVar} = @(");
        for (int i = 0; i < urls.Length; i++)
        {
            var url = urls[i];
            var isLast = (i == urls.Length - 1);
            // Add comma only if not the last item (to avoid PowerShell trailing comma error)
            var comma = isLast ? "" : ",";
            ps.AppendLine($"    {GetRandomQuote(url.Trim(), random)}{comma}");
        }
        ps.AppendLine(")");
        ps.AppendLine();

        // Loop through URLs (polymorphic foreach syntax)
        if (random.Next(2) == 0)
        {
            ps.AppendLine($"foreach ({urlVar} in ${arrayVar}) {{");
        }
        else
        {
            ps.AppendLine($"${arrayVar} | ForEach-Object {{");
            urlVar = "$_";
        }

        ps.AppendLine("    try {");
        AddRandomComment(ps, "        Executing web request", addRandomComments, random);

        // Build Invoke-WebRequest cmdlet with polymorphic parameter order
        var iwrParams = new List<string>();
        iwrParams.Add($"-Uri {urlVar}");
        iwrParams.Add($"-Method {GetRandomQuote(method, random)}");
        iwrParams.Add($"-UserAgent {GetRandomQuote(userAgent, random)}");

        if (!followRedirects)
        {
            iwrParams.Add("-MaximumRedirection 0");
        }

        // Shuffle parameters for polymorphism
        iwrParams = iwrParams.OrderBy(x => random.Next()).ToList();

        if (random.Next(2) == 0)
        {
            ps.AppendLine($"        {responseVar} = Invoke-WebRequest {string.Join(" ", iwrParams)}");
        }
        else
        {
            ps.AppendLine($"        {responseVar} = Invoke-WebRequest `");
            for (int i = 0; i < iwrParams.Count; i++)
            {
                ps.AppendLine($"            {iwrParams[i]}{(i < iwrParams.Count - 1 ? " `" : "")}");
            }
        }

        ps.AppendLine($"        {GetRandomOutputCmdlet(random)} {GetRandomQuote($"Visited: ", random)} {urlVar}");
        ps.AppendLine($"        {GetRandomOutputCmdlet(random)} {GetRandomQuote($"Status: ", random)} {responseVar}.StatusCode");

        ps.AppendLine("    }");
        ps.AppendLine("    catch {");
        ps.AppendLine($"        {GetRandomOutputCmdlet(random)} {GetRandomQuote($"Failed to visit: ", random)} {urlVar}");
        ps.AppendLine($"        {GetRandomOutputCmdlet(random)} {GetRandomQuote($"Error: ", random)} $_.Exception.Message");
        ps.AppendLine("    }");
        ps.AppendLine("}");

        return ps.ToString();
    }

    /// <summary>
    /// Generate PowerShell code for registry manipulation module
    /// </summary>
    private string GeneratePsRegistryCode(ModuleDefinition module, bool addRandomComments, Random random)
    {
        var ps = new StringBuilder();

        // Extract parameters
        var operation = module.ConfiguredValues.GetValueOrDefault("Operation", "Read")?.ToString() ?? "Read";
        var path = module.ConfiguredValues.GetValueOrDefault("Path", "HKCU:\\Software")?.ToString() ?? "HKCU:\\Software";
        var name = module.ConfiguredValues.GetValueOrDefault("Name", "")?.ToString() ?? "";
        var value = module.ConfiguredValues.GetValueOrDefault("Value", "")?.ToString() ?? "";
        var valueType = module.ConfiguredValues.GetValueOrDefault("Type", "String")?.ToString() ?? "String";

        AddRandomComment(ps, $"Registry operation: {operation}", addRandomComments, random);

        // Polymorphic variable names
        var pathVar = GetRandomVariableName(random);
        var nameVar = GetRandomVariableName(random);
        var valueVar = GetRandomVariableName(random);

        ps.AppendLine($"${pathVar} = {GetRandomQuote(path, random)}");
        if (!string.IsNullOrEmpty(name))
        {
            ps.AppendLine($"${nameVar} = {GetRandomQuote(name, random)}");
        }
        ps.AppendLine();

        ps.AppendLine("try {");

        switch (operation.ToLower())
        {
            case "read":
                ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote($"Reading registry: {path}\\{name}", random)}");
                if (string.IsNullOrEmpty(name))
                {
                    ps.AppendLine($"    ${valueVar} = Get-ItemProperty -Path ${pathVar}");
                }
                else
                {
                    ps.AppendLine($"    ${valueVar} = Get-ItemProperty -Path ${pathVar} -Name ${nameVar}");
                }
                ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote("Value: ", random)} ${valueVar}");
                break;

            case "write":
                ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote($"Writing registry: {path}\\{name}", random)}");
                ps.AppendLine($"    Set-ItemProperty -Path ${pathVar} -Name ${nameVar} -Value {GetRandomQuote(value, random)} -Type {valueType}");
                ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote("Registry value written successfully", random)}");
                break;

            case "delete":
                ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote($"Deleting registry: {path}\\{name}", random)}");
                if (string.IsNullOrEmpty(name))
                {
                    ps.AppendLine($"    Remove-Item -Path ${pathVar} -Recurse -Force");
                }
                else
                {
                    ps.AppendLine($"    Remove-ItemProperty -Path ${pathVar} -Name ${nameVar}");
                }
                ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote("Registry entry deleted", random)}");
                break;

            case "test":
                ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote($"Testing registry: {path}", random)}");
                ps.AppendLine($"    ${valueVar} = Test-Path -Path ${pathVar}");
                ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote("Exists: ", random)} ${valueVar}");
                break;
        }

        ps.AppendLine("}");
        ps.AppendLine("catch {");
        ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote("[ERROR] Registry operation failed: ", random)} $_.Exception.Message");
        ps.AppendLine("}");

        return ps.ToString();
    }

    /// <summary>
    /// Generate PowerShell code for URL visitor module (fallback)
    /// </summary>
    private string GenerateUrlVisitorCode(ModuleDefinition module, bool addRandomComments, Random random)
    {
        var ps = new StringBuilder();

        var urlsParam = module.ConfiguredValues.GetValueOrDefault("Urls", "https://www.google.com");
        var urls = urlsParam?.ToString()?.Split(',') ?? new[] { "https://www.google.com" };

        AddRandomComment(ps, "URL visitor module", addRandomComments, random);

        var urlVar = GetRandomVariableName(random);
        var responseVar = GetRandomVariableName(random);

        foreach (var url in urls)
        {
            ps.AppendLine($"${urlVar} = {GetRandomQuote(url.Trim(), random)}");
            ps.AppendLine("try {");
            ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote($"Visiting: ", random)} ${urlVar}");
            ps.AppendLine($"    ${responseVar} = Invoke-WebRequest -Uri ${urlVar} -UseBasicParsing");
            ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote($"Response length: ", random)} ${responseVar}.Content.Length");
            ps.AppendLine("}");
            ps.AppendLine("catch {");
            ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote($"Failed to visit: ", random)} ${urlVar}");
            ps.AppendLine("}");
            ps.AppendLine();
        }

        return ps.ToString();
    }

    /// <summary>
    /// Generate admin privilege check with polymorphic variations
    /// </summary>
    private void GenerateAdminCheck(StringBuilder ps, bool addRandomComments, Random random)
    {
        AddRandomComment(ps, "Administrator privilege check", addRandomComments, random);

        if (random.Next(2) == 0)
        {
            // Style 1: WindowsIdentity
            ps.AppendLine("$currentUser = [Security.Principal.WindowsIdentity]::GetCurrent()");
            ps.AppendLine("$principal = New-Object Security.Principal.WindowsPrincipal($currentUser)");
            ps.AppendLine("$adminRole = [Security.Principal.WindowsBuiltInRole]::Administrator");
            ps.AppendLine("if (-not $principal.IsInRole($adminRole)) {");
            ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote("[ERROR] This script requires administrator privileges", random)}");
            ps.AppendLine("    exit 1");
            ps.AppendLine("}");
        }
        else
        {
            // Style 2: Compact version
            ps.AppendLine("if (-not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {");
            ps.AppendLine($"    {GetRandomOutputCmdlet(random)} {GetRandomQuote("ERROR: Admin rights required", random)}");
            ps.AppendLine("    exit 1");
            ps.AppendLine("}");
        }

        ps.AppendLine();
    }

    /// <summary>
    /// Add random comment with polymorphic style
    /// </summary>
    private void AddRandomComment(StringBuilder ps, string text, bool enabled, Random random)
    {
        if (!enabled || random.Next(3) == 0) return; // 33% chance to skip

        var prefix = _commentPrefixes[random.Next(_commentPrefixes.Length)];

        if (random.Next(3) == 0)
        {
            // Block comment style
            ps.AppendLine("<#");
            ps.AppendLine($"    {text}");
            ps.AppendLine("#>");
        }
        else
        {
            // Single line comment
            ps.AppendLine($"{prefix}{text}");
        }
    }

    /// <summary>
    /// Add random whitespace for polymorphism
    /// </summary>
    private void AddRandomWhitespace(StringBuilder ps, Random random)
    {
        var lines = random.Next(0, 3);
        for (int i = 0; i < lines; i++)
        {
            ps.AppendLine();
        }
    }

    /// <summary>
    /// Get random variable name
    /// </summary>
    private string GetRandomVariableName(Random random)
    {
        var styleIndex = random.Next(_variableStyles.Length);
        var style = _variableStyles[styleIndex];
        var baseName = style[random.Next(style.Length)];
        var suffix = random.Next(1, 1000);
        return $"{baseName}{suffix}";
    }

    /// <summary>
    /// Get random output cmdlet
    /// </summary>
    private string GetRandomOutputCmdlet(Random random)
    {
        return _outputCmdlets[random.Next(_outputCmdlets.Length)];
    }

    /// <summary>
    /// Wrap string in random quote style
    /// </summary>
    private string GetRandomQuote(string text, Random random)
    {
        // 50/50 single vs double quotes
        return random.Next(2) == 0
            ? $"'{text}'"
            : $"\"{text}\"";
    }
}
