/*
 * EXECUTION TYPE: PDF Document
 * ID: pdf-document
 * DESCRIPTION: Generates PDF documents with content from configured modules only
 * CATEGORY: Document
 * FILE EXTENSION: .pdf
 *
 * PARAMETERS:
 * @Title:string:Document:::PDF document title
 * @Subject:string::::PDF subject line (optional)
 * @PageCount:int:1:::Number of pages to generate
 * @JavaScriptTitle:string:Alert:::Title for JavaScript popups (if pdf-javascript module used)
 *
 * SUPPORTED MODULES:
 * - pdf-text: Adds text content blocks
 * - pdf-hyperlink: Adds clickable hyperlinks
 * - pdf-image: Embeds images
 * - pdf-javascript: Adds JavaScript that executes on document open/close/print
 *
 * HOW IT WORKS:
 * Generates PDF content PURELY from configured modules:
 * - Add pdf-text modules for text blocks
 * - Add pdf-hyperlink modules for clickable links
 * - Add pdf-image modules for embedded images
 * - Add pdf-javascript module for interactive actions
 * - If no modules are configured, generates a minimal blank PDF
 *
 * USE CASE:
 * Test phishing detection and PDF analysis in EDR/email security
 */

using System;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Collections.Generic;
using WildfireBuffet.Core.Abstractions;
using WildfireBuffet.Core.Models;
using WildfireBuffet.Core.Helpers;
using QuestPDF.Fluent;
using QuestPDF.Helpers;
using QuestPDF.Infrastructure;
using PdfSharp.Pdf;
using PdfSharp.Pdf.IO;

public class PdfDocument : ExecutionTypeBase
{
    public override ExecutionTypeDescriptor Descriptor => new()
    {
        Id = "pdf-document",
        Name = "PDF Document",
        Description = "PDF file generated purely from configured modules (pdf-text, pdf-hyperlink, pdf-image, pdf-javascript)",
        FileExtension = ".pdf",
        Icon = "📄",
        Category = "Document",
        SupportedModules = new[] { "pdf-header", "pdf-footer", "pdf-text", "pdf-hyperlink", "pdf-image", "pdf-javascript", "url-visitor" },
        ConfigurableParameters = new Dictionary<string, string>
        {
            { "Title", "string - PDF document title" },
            { "Subject", "string - PDF subject line (optional)" },
            { "PageCount", "int - Number of pages to generate" },
            { "JavaScriptTitle", "string - Title for JavaScript popup (if pdf-javascript module used)" }
        }
    };

    public override async Task<ExecutionResult> GenerateAsync(ExecutionContext context)
    {
        // Validate module compatibility
        var validationError = ValidateModuleCompatibility(context);
        if (validationError != null)
        {
            return validationError;
        }

        // Get configuration
        var title = GetConfig(context, "Title", "Document");
        var subject = GetConfig(context, "Subject", "");
        var pageCount = GetConfig(context, "PageCount", 1);
        var jsTitle = GetConfig(context, "JavaScriptTitle", "Alert");

        try
        {
            // Configure QuestPDF license (Community license for testing)
            QuestPDF.Settings.License = LicenseType.Community;

            // Generate PDF bytes
            var pdfBytes = await Task.Run(() => GeneratePdfWithQuestPDF(context, title, subject, pageCount, jsTitle));

            var result = ExecutionResult.Succeeded(pdfBytes);
            result.Diagnostics.Add($"Generated {pageCount} page PDF with {context.Modules.Count} module(s)");
            result.Diagnostics.Add($"Title: {title}, Subject: {subject}");

            return result;
        }
        catch (Exception ex)
        {
            return ExecutionResult.Failed($"PDF generation failed: {ex.Message}\n{ex.StackTrace}");
        }
    }

    private byte[] GeneratePdfWithQuestPDF(ExecutionContext context, string title, string subject, int pageCount, string jsTitle)
    {
        // Polymorphic randomization based on sample number
        var random = CreatePolymorphicRandom(context.SampleNumber);

        // Extract URLs from modules
        var urls = new List<string>();
        foreach (var module in context.Modules)
        {
            if (module.Id == "url-visitor" && module.ConfiguredValues.TryGetValue("Url", out var urlObj) && urlObj is string url)
            {
                urls.Add(url);
            }
        }

        var document = Document.Create(container =>
        {
            // Create multiple pages
            for (int pageNum = 1; pageNum <= pageCount; pageNum++)
            {
                container.Page(page =>
                {
                    page.Size(PageSizes.A4);

                    // Polymorphic margins (slight variations)
                    var baseMargin = 40;
                    var marginVariation = random.Next(-5, 6); // ±5 points
                    page.Margin(baseMargin + marginVariation);

                    page.PageColor(Colors.White);

                    // Polymorphic font size (very slight variation)
                    var baseFontSize = 12;
                    var fontVariation = random.Next(0, 2) * 0.5f; // 0, 0.5, or 1
                    page.DefaultTextStyle(x => x.FontSize(baseFontSize + fontVariation).FontFamily("Arial"));

                    // Header - only if pdf-header module is present
                    var headerModule = context.Modules.FirstOrDefault(m => m.Id == "pdf-header");
                    if (headerModule != null)
                    {
                        page.Header().Element(container => ComposeHeader(container, headerModule));
                    }

                    // Content for this specific page
                    page.Content().Element(cont => ComposePageContent(cont, context, urls, pageNum, pageCount));

                    // Footer - only if pdf-footer module is present
                    var footerModule = context.Modules.FirstOrDefault(m => m.Id == "pdf-footer");
                    if (footerModule != null)
                    {
                        page.Footer().Element(container => ComposeFooter(container, footerModule, pageNum, pageCount));
                    }
                });
            }
        });

        // Generate PDF bytes
        var pdfBytes = document.GeneratePdf();

        // Always apply polymorphic modifications using PdfSharp to ensure unique hashes
        // This ensures that each sample has a different hash even without JavaScript
        pdfBytes = ApplyPolymorphicModifications(pdfBytes, context.SampleNumber);

        // Add JavaScript only if pdf-javascript module is present
        var jsModule = context.Modules.FirstOrDefault(m => m.Id == "pdf-javascript");
        if (jsModule != null)
        {
            var jsCode = jsModule.ConfiguredValues.TryGetValue("JsCode", out var jc) ? jc?.ToString() : "";
            var jsTrigger = jsModule.ConfiguredValues.TryGetValue("TriggerEvent", out var jt) ? jt?.ToString() : "onOpen";

            // Use configured JavaScript from module
            var message = !string.IsNullOrEmpty(jsCode) ? jsCode : "alert('Document opened');";
            var useAutoOpen = jsTrigger == "onOpen";

            pdfBytes = AddJavaScriptToPdf(pdfBytes, message, jsTitle, useAutoOpen, urls.FirstOrDefault());
        }

        return pdfBytes;

        void ComposeHeader(IContainer container, ModuleDefinition headerModule)
        {
            // Extract configuration from pdf-header module
            var headerTitle = headerModule.ConfiguredValues.TryGetValue("Title", out var ht) ? ht?.ToString() : "Document";
            var subtitle = headerModule.ConfiguredValues.TryGetValue("Subtitle", out var st) ? st?.ToString() : "";
            var showLine = headerModule.ConfiguredValues.TryGetValue("ShowLine", out var sl) && bool.TryParse(sl?.ToString(), out var slBool) && slBool;
            var fontSize = headerModule.ConfiguredValues.TryGetValue("FontSize", out var fs) && int.TryParse(fs?.ToString(), out var fsInt) ? fsInt : 24;

            container.Column(column =>
            {
                column.Item().Text(headerTitle).FontSize(fontSize).Bold().FontColor(Colors.Blue.Darken2);

                if (!string.IsNullOrWhiteSpace(subtitle))
                {
                    column.Item().PaddingTop(5).Text(subtitle).FontSize(14).Italic();
                }

                if (showLine)
                {
                    column.Item().PaddingTop(10).LineHorizontal(1).LineColor(Colors.Grey.Lighten2);
                }
            });
        }

        void ComposePageContent(IContainer container, ExecutionContext ctx, List<string> linkUrls, int pageNum, int totalPages)
        {
            container.Column(column =>
            {
                // Add invisible polymorphic content (white text on white background)
                var invisibleText = GetPolymorphicInvisibleText(ctx.SampleNumber);
                column.Item().Text(invisibleText).FontSize(1).FontColor(Colors.White);

                // Only show page indicator if multiple pages
                if (totalPages > 1)
                {
                    column.Item().PaddingTop(20).Text($"Page {pageNum} of {totalPages}").FontSize(10).Italic();
                }

                // Add module content (only on first page)
                if (pageNum == 1)
                {
                    var hasModules = ctx.Modules.Any();

                    if (!hasModules)
                    {
                        // If no modules configured, show minimal placeholder
                        column.Item().PaddingTop(40).AlignCenter().Text(title).FontSize(18).Bold();
                    }
                    else
                    {
                        // Generate content ONLY from configured modules
                        foreach (var module in ctx.Modules)
                        {
                            switch (module.Id)
                            {
                                case "pdf-text":
                                    var textContent = module.ConfiguredValues.TryGetValue("TextContent", out var tc) ? tc?.ToString() : "";
                                    var fontSize = module.ConfiguredValues.TryGetValue("FontSize", out var fs) && int.TryParse(fs?.ToString(), out var fsInt) ? fsInt : 14;
                                    var isBold = module.ConfiguredValues.TryGetValue("IsBold", out var ib) && bool.TryParse(ib?.ToString(), out var ibBool) && ibBool;

                                    if (!string.IsNullOrEmpty(textContent))
                                    {
                                        column.Item().PaddingTop(15).Text(text =>
                                        {
                                            var span = text.Span(textContent).FontSize(fontSize);
                                            if (isBold) span.Bold();
                                        });
                                    }
                                    break;

                                case "pdf-hyperlink":
                                    var linkText = module.ConfiguredValues.TryGetValue("LinkText", out var lt) ? lt?.ToString() : "";
                                    var linkUrl = module.ConfiguredValues.TryGetValue("LinkUrl", out var lu) ? lu?.ToString() : "";
                                    var position = module.ConfiguredValues.TryGetValue("Position", out var pos) ? pos?.ToString() : "center";

                                    if (!string.IsNullOrEmpty(linkText) && !string.IsNullOrEmpty(linkUrl))
                                    {
                                        column.Item().PaddingTop(15).Column(linkCol =>
                                        {
                                            var item = position?.ToLower() switch
                                            {
                                                "top" => linkCol.Item().AlignTop(),
                                                "bottom" => linkCol.Item().AlignBottom(),
                                                _ => linkCol.Item().AlignCenter()
                                            };

                                            item.Hyperlink(linkUrl)
                                                .Text(linkText)
                                                .FontSize(12)
                                                .FontColor(Colors.Blue.Medium)
                                                .Underline();
                                        });
                                    }
                                    break;

                                case "pdf-image":
                                    var imagePath = module.ConfiguredValues.TryGetValue("ImagePath", out var ip) ? ip?.ToString() : "";
                                    var imagePosition = module.ConfiguredValues.TryGetValue("Position", out var imgPos) ? imgPos?.ToString() : "center";
                                    var imageWidth = module.ConfiguredValues.TryGetValue("Width", out var iw) && int.TryParse(iw?.ToString(), out var iwInt) ? iwInt : 200;

                                    if (!string.IsNullOrEmpty(imagePath) && File.Exists(imagePath))
                                    {
                                        column.Item().PaddingTop(15).Column(imgCol =>
                                        {
                                            var item = imagePosition?.ToLower() switch
                                            {
                                                "top" => imgCol.Item().AlignTop(),
                                                "bottom" => imgCol.Item().AlignBottom(),
                                                _ => imgCol.Item().AlignCenter()
                                            };

                                            item.Image(imagePath).FitWidth();
                                        });
                                    }
                                    break;

                                case "pdf-javascript":
                                    // JavaScript is handled separately in AddJavaScriptToPdf method
                                    // No visual content needed in the PDF body
                                    break;

                                case "url-visitor":
                                    // Legacy support: url-visitor module becomes a clickable link
                                    if (module.ConfiguredValues.TryGetValue("Url", out var urlObj) && urlObj is string url && !string.IsNullOrEmpty(url))
                                    {
                                        column.Item().PaddingTop(15).AlignCenter()
                                            .Hyperlink(url)
                                            .Text(url)
                                            .FontSize(12)
                                            .FontColor(Colors.Blue.Medium)
                                            .Underline();
                                    }
                                    break;
                            }
                        }
                    }
                }

                // No filler content - PDF contains ONLY module content
            });
        }

        void ComposeFooter(IContainer container, ModuleDefinition footerModule, int pageNum, int totalPages)
        {
            // Extract configuration from pdf-footer module
            var footerText = footerModule.ConfiguredValues.TryGetValue("Text", out var ft) ? ft?.ToString() : "";
            var showPageNumber = footerModule.ConfiguredValues.TryGetValue("ShowPageNumber", out var spn) && bool.TryParse(spn?.ToString(), out var spnBool) && spnBool;
            var fontSize = footerModule.ConfiguredValues.TryGetValue("FontSize", out var fs) && int.TryParse(fs?.ToString(), out var fsInt) ? fsInt : 10;

            container.Column(column =>
            {
                // Show page number if enabled
                if (showPageNumber)
                {
                    column.Item().AlignCenter().Text(text =>
                    {
                        text.Span("Page ").FontSize(fontSize);
                        text.Span($"{pageNum}").FontSize(fontSize).Bold();
                        text.Span($" of {totalPages}").FontSize(fontSize);
                    });
                }

                // Show custom footer text if provided
                if (!string.IsNullOrWhiteSpace(footerText))
                {
                    column.Item().AlignCenter().Text(footerText).FontSize(fontSize).FontColor(Colors.Grey.Medium);
                }
            });
        }
    }


    /// <summary>
    /// Apply polymorphic modifications to PDF using PdfSharp to ensure unique hashes.
    /// This adds sample-specific metadata that survives PDF serialization.
    /// </summary>
    private byte[] ApplyPolymorphicModifications(byte[] pdfBytes, int sampleNumber)
    {
        try
        {
            using var ms = new MemoryStream(pdfBytes);
            using var pdfDoc = PdfReader.Open(ms, PdfDocumentOpenMode.Modify);

            // Add polymorphic metadata (survives serialization)
            var random = PolymorphismHelper.CreateRandom(sampleNumber);

            // Polymorphic author
            pdfDoc.Info.Author = GetPolymorphicAuthor(sampleNumber);

            // Polymorphic creator
            pdfDoc.Info.Creator = GetPolymorphicCreator(sampleNumber);

            // Note: Producer property is read-only in PdfSharp, so we can't set it

            // Polymorphic keywords
            var keywords = new[]
            {
                "document", "invoice", "report", "statement", "notice",
                "information", "correspondence", "communication", "record", "file"
            };

            var selectedKeywords = new List<string>();
            for (int i = 0; i < 3; i++)
            {
                selectedKeywords.Add(keywords[(sampleNumber + i) % keywords.Length]);
            }

            // Add deterministic random suffix based on sample number
            var suffix = random.Next(10000000, 99999999).ToString("x8");
            pdfDoc.Info.Keywords = string.Join(", ", selectedKeywords) + $" - {suffix}";

            // Add polymorphic custom metadata to subject
            var currentSubject = pdfDoc.Info.Subject ?? "";
            pdfDoc.Info.Subject = currentSubject + $" [{random.Next(10000, 99999)}]";

            // Polymorphic creation date
            pdfDoc.Info.CreationDate = DateTime.UtcNow.AddSeconds(random.Next(-3600, 3600));

            using var outMs = new MemoryStream();
            pdfDoc.Save(outMs, false);
            return outMs.ToArray();
        }
        catch (Exception ex)
        {
            // If modification fails, return original PDF
            Console.WriteLine($"[Warning] Polymorphic modification failed: {ex.Message}");
            return pdfBytes;
        }
    }

    private byte[] AddJavaScriptToPdf(byte[] pdfBytes, string message, string title, bool autoOpen, string? url)
    {
        try
        {
            using var ms = new MemoryStream(pdfBytes);
            using var pdfDoc = PdfReader.Open(ms, PdfDocumentOpenMode.Modify);

            // Create JavaScript to show popup
            var jsCode = new StringBuilder();
            jsCode.AppendLine("app.alert({");
            jsCode.AppendLine($"  cMsg: '{message.Replace("'", "\\'")}',");
            jsCode.AppendLine($"  cTitle: '{title.Replace("'", "\\'")}',");
            jsCode.AppendLine("  nIcon: 2,");  // Warning icon
            jsCode.AppendLine("  nType: 0");    // OK button
            jsCode.AppendLine("});");

            // If autoOpen is enabled and URL provided, launch URL after alert
            if (autoOpen && !string.IsNullOrEmpty(url))
            {
                jsCode.AppendLine($"app.launchURL('{url}', true);");
            }

            // Create JavaScript action dictionary
            var jsAction = new PdfDictionary(pdfDoc);
            jsAction.Elements.Add("/S", new PdfName("/JavaScript"));
            jsAction.Elements.Add("/JS", new PdfString(jsCode.ToString()));

            // Add OpenAction to catalog
            pdfDoc.Internals.Catalog.Elements["/OpenAction"] = jsAction;

            using var outMs = new MemoryStream();
            pdfDoc.Save(outMs, false);
            return outMs.ToArray();
        }
        catch (Exception ex)
        {
            // If JavaScript injection fails, return original PDF
            Console.WriteLine($"[Warning] JavaScript injection failed: {ex.Message}");
            return pdfBytes;
        }
    }

    private string GetPolymorphicAuthor(int sampleNumber)
    {
        var authors = new[]
        {
            "John Smith", "Mary Johnson", "Robert Williams", "Jennifer Brown",
            "Michael Davis", "Linda Miller", "William Wilson", "Elizabeth Moore",
            "David Taylor", "Barbara Anderson", "Richard Thomas", "Susan Jackson"
        };
        return authors[sampleNumber % authors.Length];
    }

    private string GetPolymorphicCreator(int sampleNumber)
    {
        var creators = new[]
        {
            "Microsoft Word 16.0", "Adobe Acrobat 23.1", "LibreOffice 7.5",
            "Microsoft Word 15.0", "Adobe Acrobat 22.3", "WPS Office 11.2",
            "Microsoft Word 16.5", "Adobe Acrobat DC", "OpenOffice 4.1",
            "Google Docs", "Apple Pages 12.2", "Foxit PhantomPDF 11.0"
        };
        return creators[sampleNumber % creators.Length];
    }

    private string GetPolymorphicProducer(int sampleNumber)
    {
        var producers = new[]
        {
            "Microsoft: Print To PDF", "Adobe PDF Library 23.1", "GPL Ghostscript 10.0",
            "Skia/PDF m116", "iText 7.2.5", "PDFsharp 1.51",
            "wkhtmltopdf 0.12.6", "Prince 15.1", "WeasyPrint 60.0",
            "pdfTeX-1.40.25", "Apache FOP 2.8", "Chromium"
        };
        return producers[sampleNumber % producers.Length];
    }

    private string GetPolymorphicInvisibleText(int sampleNumber)
    {
        var random = PolymorphismHelper.CreateRandom(sampleNumber);
        var phrases = new[]
        {
            "This document contains sensitive information",
            "Confidential business communication",
            "Internal use only - do not distribute",
            "Property of the organization",
            "For authorized personnel only",
            "Trade secret material enclosed",
            "Privileged and confidential",
            "Attorney work product",
            "Protected health information",
            "Proprietary data included"
        };

        // Select 2-4 random phrases and join them with spaces
        var count = random.Next(2, 5);
        var selectedPhrases = new List<string>();
        for (int i = 0; i < count; i++)
        {
            selectedPhrases.Add(phrases[(sampleNumber + i) % phrases.Length]);
        }

        return string.Join(" ", selectedPhrases);
    }
}
