#!/usr/bin/env pwsh

Write-Host "Verifying Multi-DLL exports..." -ForegroundColor Cyan
Write-Host ""

$dllExports = @{
    'kernel32.dll' = @(
        'GetComputerNameW',
        'GetLocalTime',
        'GetDiskFreeSpaceW',
        'GetModuleFileNameW',
        'LoadLibraryW',
        'GetProcAddress'
    )
    'advapi32.dll' = @(
        'GetUserNameW'
    )
}

Add-Type -TypeDefinition @"
using System;
using System.Runtime.InteropServices;

public class DllExportChecker {
    [DllImport("kernel32.dll", CharSet = CharSet.Unicode)]
    public static extern IntPtr LoadLibrary(string lpFileName);

    [DllImport("kernel32.dll", CharSet = CharSet.Ansi)]
    public static extern IntPtr GetProcAddress(IntPtr hModule, string lpProcName);
}
"@

$totalFound = 0
$totalNotFound = 0

foreach ($dll in $dllExports.Keys) {
    Write-Host "Checking $dll..." -ForegroundColor Yellow

    $handle = [DllExportChecker]::LoadLibrary($dll)

    if ($handle -eq [IntPtr]::Zero) {
        Write-Host "  [ERROR] Could not load $dll" -ForegroundColor Red
        continue
    }

    Write-Host "  [OK] Loaded $dll at 0x$($handle.ToString('X'))" -ForegroundColor Green

    foreach ($api in $dllExports[$dll]) {
        $addr = [DllExportChecker]::GetProcAddress($handle, $api)
        if ($addr -ne [IntPtr]::Zero) {
            Write-Host "    [OK] $api at 0x$($addr.ToString('X'))" -ForegroundColor Green
            $totalFound++
        } else {
            Write-Host "    [FAIL] $api NOT FOUND" -ForegroundColor Red
            $totalNotFound++
        }
    }

    Write-Host ""
}

Write-Host "Summary:" -ForegroundColor Cyan
Write-Host "  Found: $totalFound" -ForegroundColor Green
Write-Host "  Not Found: $totalNotFound" -ForegroundColor $(if ($totalNotFound -gt 0) { "Red" } else { "Green" })

if ($totalNotFound -gt 0) {
    exit 1
}
