#!/usr/bin/env pwsh
# Test all recipes using Docker with volume mount for output

$outputDir = "X:\Projects\dotnet\wildfire-buffet\WildfireBuffet.UI\bin\Release\net8.0\win-x64\output"
$recipesDir = "X:\Projects\dotnet\wildfire-buffet\Recipes"
$dockerImage = "wildfire-buffet-cli:latest"

Write-Host "`n=== Testing All Recipes with Docker ===" -ForegroundColor Cyan
Write-Host "Docker Image: $dockerImage" -ForegroundColor Yellow
Write-Host "Output Directory: $outputDir" -ForegroundColor Yellow
Write-Host "Note: Docker will mount the output directory as a volume`n" -ForegroundColor Gray

# Ensure output directory exists
if (-not (Test-Path $outputDir)) {
    New-Item -ItemType Directory -Path $outputDir -Force | Out-Null
    Write-Host "[+] Created output directory: $outputDir`n" -ForegroundColor Green
}

# Get all recipe files
$recipes = Get-ChildItem "$recipesDir\*.json" | Sort-Object Name

$results = @()

foreach ($recipe in $recipes) {
    Write-Host "`n[$($recipe.Name)]" -ForegroundColor Green -NoNewline
    Write-Host " Building with Docker..." -NoNewline

    # Docker run command with volume mounts
    # Mount recipes directory (read-only) and output directory (read-write)
    $dockerCmd = @(
        "run", "--rm",
        "-v", "${recipesDir}:C:\wildfire-buffet\Recipes:ro",
        "-v", "${outputDir}:C:\wildfire-buffet\output",
        $dockerImage,
        "--config", "Recipes\$($recipe.Name)",
        "--output", "C:\wildfire-buffet\output"
    )

    try {
        $output = docker @dockerCmd 2>&1 | Out-String

        if ($output -match "1 succeeded" -or $output -match "Success") {
            Write-Host " [SUCCESS]" -ForegroundColor Green
            $results += @{ Recipe = $recipe.Name; Status = "SUCCESS" }
        }
        elseif ($output -match "Failed|Error") {
            Write-Host " [FAILED]" -ForegroundColor Red
            $error = ($output -split "`n" | Select-String -Pattern "Error:|Failed:").Line | Select-Object -First 1
            Write-Host "  Error: $error" -ForegroundColor DarkRed
            $results += @{ Recipe = $recipe.Name; Status = "FAILED"; Error = $error }
        }
        else {
            Write-Host " [UNKNOWN]" -ForegroundColor Yellow
            Write-Host "  Output: $($output.Substring(0, [Math]::Min(200, $output.Length)))" -ForegroundColor DarkYellow
            $results += @{ Recipe = $recipe.Name; Status = "UNKNOWN" }
        }
    }
    catch {
        Write-Host " [ERROR]" -ForegroundColor Red
        Write-Host "  Exception: $($_.Exception.Message)" -ForegroundColor DarkRed
        $results += @{ Recipe = $recipe.Name; Status = "ERROR"; Error = $_.Exception.Message }
    }
}

Write-Host "`n`n=== Summary ===" -ForegroundColor Cyan
$successCount = ($results | Where-Object { $_.Status -eq "SUCCESS" }).Count
$failedCount = ($results | Where-Object { $_.Status -eq "FAILED" }).Count
$unknownCount = ($results | Where-Object { $_.Status -eq "UNKNOWN" }).Count
$errorCount = ($results | Where-Object { $_.Status -eq "ERROR" }).Count

Write-Host "Total Recipes: $($recipes.Count)" -ForegroundColor White
Write-Host "[+] Succeeded: $successCount" -ForegroundColor Green
Write-Host "[-] Failed: $failedCount" -ForegroundColor Red
Write-Host "[?] Unknown: $unknownCount" -ForegroundColor Yellow
Write-Host "[!] Errors: $errorCount" -ForegroundColor Magenta

# Show output directory contents
Write-Host "`n`n=== Output Directory Contents ===" -ForegroundColor Cyan
if (Test-Path $outputDir) {
    $files = Get-ChildItem $outputDir | Sort-Object LastWriteTime -Descending | Select-Object -First 20
    Write-Host "Showing most recent 20 files:" -ForegroundColor Gray
    foreach ($file in $files) {
        $size = if ($file.Length -gt 1MB) { "{0:N2} MB" -f ($file.Length / 1MB) }
                elseif ($file.Length -gt 1KB) { "{0:N2} KB" -f ($file.Length / 1KB) }
                else { "$($file.Length) B" }
        $age = (Get-Date) - $file.LastWriteTime
        $ageStr = if ($age.TotalMinutes -lt 1) { "just now" }
                  elseif ($age.TotalMinutes -lt 60) { "{0:N0} min ago" -f $age.TotalMinutes }
                  elseif ($age.TotalHours -lt 24) { "{0:N0} hr ago" -f $age.TotalHours }
                  else { "{0:N0} days ago" -f $age.TotalDays }
        Write-Host "  $($file.Name) - $size ($ageStr)" -ForegroundColor Gray
    }
    Write-Host "`nTotal files in output: $((Get-ChildItem $outputDir).Count)" -ForegroundColor White
}
else {
    Write-Host "  Output directory not found!" -ForegroundColor Red
}
