<#
.SYNOPSIS
    Analyzes PE (Portable Executable) files and extracts import table information.

.DESCRIPTION
    This script reads PE files (EXE/DLL) and parses their import tables to show
    which functions are statically imported. This is useful for verifying dynamic
    loading behavior - if a function is NOT in the import table but is used at
    runtime, it must be dynamically loaded via LoadLibrary/GetProcAddress.

.PARAMETER FilePath
    Path to the PE file (EXE or DLL) to analyze.

.PARAMETER SearchFunctions
    Optional array of function names to specifically search for in imports.

.PARAMETER Detailed
    Show detailed information including DLL names and all imported functions.

.EXAMPLE
    .\Get-PEImports.ps1 -FilePath "payload.exe"
    Shows all imported DLLs and their function counts.

.EXAMPLE
    .\Get-PEImports.ps1 -FilePath "payload.exe" -SearchFunctions @("GetSystemInfo", "LoadLibrary")
    Checks if specific functions are in the static import table.

.EXAMPLE
    .\Get-PEImports.ps1 -FilePath "payload.exe" -Detailed
    Shows all DLLs and every imported function.

.NOTES
    Author: ril3y
    This script does not require external tools like dumpbin.exe.
    It directly parses the PE file format using PowerShell.
#>

[CmdletBinding()]
param(
    [Parameter(Mandatory = $true, Position = 0)]
    [string]$FilePath,

    [Parameter(Mandatory = $false)]
    [string[]]$SearchFunctions,

    [Parameter(Mandatory = $false)]
    [switch]$Detailed
)

function Read-PEImportTable {
    param(
        [byte[]]$FileBytes
    )

    $imports = @{}

    try {
        # Check DOS header signature (MZ)
        $dosSignature = [System.Text.Encoding]::ASCII.GetString($FileBytes[0..1])
        if ($dosSignature -ne "MZ") {
            throw "Invalid PE file: Missing MZ signature"
        }

        # Get PE header offset (at offset 0x3C)
        $peOffset = [System.BitConverter]::ToInt32($FileBytes, 0x3C)

        # Check PE signature
        $peSignature = [System.Text.Encoding]::ASCII.GetString($FileBytes[$peOffset..($peOffset + 3)])
        if ($peSignature -ne "PE`0`0") {
            throw "Invalid PE file: Missing PE signature"
        }

        # Get machine type (offset +4 from PE signature)
        $machineType = [System.BitConverter]::ToUInt16($FileBytes, $peOffset + 4)
        $is64bit = ($machineType -eq 0x8664)  # IMAGE_FILE_MACHINE_AMD64

        # Get optional header offset
        $optHeaderOffset = $peOffset + 24

        # Get data directory offset (different for 32-bit vs 64-bit)
        if ($is64bit) {
            $dataDirectoryOffset = $optHeaderOffset + 112
        } else {
            $dataDirectoryOffset = $optHeaderOffset + 96
        }

        # Import table is the second data directory entry (index 1)
        $importTableRVA = [System.BitConverter]::ToUInt32($FileBytes, $dataDirectoryOffset + 8)
        $importTableSize = [System.BitConverter]::ToUInt32($FileBytes, $dataDirectoryOffset + 12)

        if ($importTableRVA -eq 0) {
            Write-Warning "No import table found in PE file"
            return $imports
        }

        # Convert RVA to file offset (simplified - assumes imports are in first section)
        $sectionHeaderOffset = $optHeaderOffset +
            [System.BitConverter]::ToUInt16($FileBytes, $peOffset + 20)

        # Read section headers to find import table location
        $numberOfSections = [System.BitConverter]::ToUInt16($FileBytes, $peOffset + 6)
        $importTableOffset = 0

        for ($i = 0; $i -lt $numberOfSections; $i++) {
            $sectionOffset = $sectionHeaderOffset + ($i * 40)
            $virtualAddress = [System.BitConverter]::ToUInt32($FileBytes, $sectionOffset + 12)
            $rawDataPtr = [System.BitConverter]::ToUInt32($FileBytes, $sectionOffset + 20)
            $virtualSize = [System.BitConverter]::ToUInt32($FileBytes, $sectionOffset + 8)

            if ($importTableRVA -ge $virtualAddress -and
                $importTableRVA -lt ($virtualAddress + $virtualSize)) {
                $importTableOffset = $rawDataPtr + ($importTableRVA - $virtualAddress)
                break
            }
        }

        if ($importTableOffset -eq 0) {
            Write-Warning "Could not locate import table in file sections"
            return $imports
        }

        # Parse import descriptors
        $offset = $importTableOffset
        while ($true) {
            # Read import descriptor (20 bytes)
            $originalFirstThunk = [System.BitConverter]::ToUInt32($FileBytes, $offset)
            $timeDateStamp = [System.BitConverter]::ToUInt32($FileBytes, $offset + 4)
            $forwarderChain = [System.BitConverter]::ToUInt32($FileBytes, $offset + 8)
            $nameRVA = [System.BitConverter]::ToUInt32($FileBytes, $offset + 12)
            $firstThunk = [System.BitConverter]::ToUInt32($FileBytes, $offset + 16)

            # End of import descriptors (null descriptor)
            if ($nameRVA -eq 0) {
                break
            }

            # Convert DLL name RVA to file offset and read DLL name
            $dllNameOffset = ConvertRVAToOffset $nameRVA $sectionHeaderOffset $numberOfSections $FileBytes
            $dllName = Read-NullTerminatedString $FileBytes $dllNameOffset

            # Read imported functions
            $thunkRVA = if ($originalFirstThunk -ne 0) { $originalFirstThunk } else { $firstThunk }
            $thunkOffset = ConvertRVAToOffset $thunkRVA $sectionHeaderOffset $numberOfSections $FileBytes

            $functions = @()
            $thunkPos = $thunkOffset

            while ($true) {
                if ($is64bit) {
                    $thunkValue = [System.BitConverter]::ToUInt64($FileBytes, $thunkPos)
                    $thunkPos += 8
                } else {
                    $thunkValue = [System.BitConverter]::ToUInt32($FileBytes, $thunkPos)
                    $thunkPos += 4
                }

                if ($thunkValue -eq 0) {
                    break
                }

                # Check if import by ordinal
                $ordinalFlag = if ($is64bit) { 0x8000000000000000 } else { 0x80000000 }
                if (($thunkValue -band $ordinalFlag) -ne 0) {
                    $ordinal = $thunkValue -band 0xFFFF
                    $functions += "Ordinal_$ordinal"
                } else {
                    # Import by name
                    $nameTableRVA = $thunkValue
                    $nameTableOffset = ConvertRVAToOffset $nameTableRVA $sectionHeaderOffset $numberOfSections $FileBytes
                    # Skip hint (2 bytes) and read function name
                    $functionName = Read-NullTerminatedString $FileBytes ($nameTableOffset + 2)
                    $functions += $functionName
                }
            }

            $imports[$dllName] = $functions
            $offset += 20
        }

    } catch {
        Write-Error "Error parsing PE file: $_"
        Write-Error $_.ScriptStackTrace
    }

    return $imports
}

function ConvertRVAToOffset {
    param(
        [uint32]$RVA,
        [int]$SectionHeaderOffset,
        [int]$NumberOfSections,
        [byte[]]$FileBytes
    )

    for ($i = 0; $i -lt $NumberOfSections; $i++) {
        $sectionOffset = $SectionHeaderOffset + ($i * 40)
        $virtualAddress = [System.BitConverter]::ToUInt32($FileBytes, $sectionOffset + 12)
        $rawDataPtr = [System.BitConverter]::ToUInt32($FileBytes, $sectionOffset + 20)
        $virtualSize = [System.BitConverter]::ToUInt32($FileBytes, $sectionOffset + 8)

        if ($RVA -ge $virtualAddress -and $RVA -lt ($virtualAddress + $virtualSize)) {
            return $rawDataPtr + ($RVA - $virtualAddress)
        }
    }

    return 0
}

function Read-NullTerminatedString {
    param(
        [byte[]]$Bytes,
        [int]$Offset
    )

    $sb = New-Object System.Text.StringBuilder
    $pos = $Offset

    while ($pos -lt $Bytes.Length) {
        $byte = $Bytes[$pos]
        if ($byte -eq 0) {
            break
        }
        [void]$sb.Append([char]$byte)
        $pos++
    }

    return $sb.ToString()
}

# Main script execution
Write-Host "`n=== PE Import Table Analyzer ===" -ForegroundColor Cyan
Write-Host "File: $FilePath`n" -ForegroundColor Yellow

if (-not (Test-Path $FilePath)) {
    Write-Error "File not found: $FilePath"
    exit 1
}

# Read file
$fileBytes = [System.IO.File]::ReadAllBytes($FilePath)
$fileSize = $fileBytes.Length
Write-Host "File Size: $($fileSize.ToString('N0')) bytes`n"

# Parse imports
$imports = Read-PEImportTable -FileBytes $fileBytes

if ($imports.Count -eq 0) {
    Write-Host "No imports found (or failed to parse import table)" -ForegroundColor Yellow
    exit 0
}

# Display results
Write-Host "Found imports from $($imports.Count) DLL(s):`n" -ForegroundColor Green

if ($SearchFunctions) {
    Write-Host "Searching for specific functions..." -ForegroundColor Cyan
    $found = @{}
    $notFound = @()

    foreach ($funcName in $SearchFunctions) {
        $foundInDll = $null
        foreach ($dll in $imports.Keys) {
            if ($imports[$dll] -contains $funcName) {
                $foundInDll = $dll
                break
            }
        }

        if ($foundInDll) {
            if (-not $found.ContainsKey($foundInDll)) {
                $found[$foundInDll] = @()
            }
            $found[$foundInDll] += $funcName
        } else {
            $notFound += $funcName
        }
    }

    if ($found.Count -gt 0) {
        Write-Host "`n[FOUND] Statically imported functions:" -ForegroundColor Green
        foreach ($dll in $found.Keys) {
            Write-Host "  From $dll" -ForegroundColor Yellow
            foreach ($func in $found[$dll]) {
                Write-Host "    [+] $func" -ForegroundColor Green
            }
        }
    }

    if ($notFound.Count -gt 0) {
        Write-Host "`n[NOT FOUND] These functions are NOT statically imported:" -ForegroundColor Cyan
        Write-Host "  (They may be dynamically loaded via LoadLibrary/GetProcAddress)" -ForegroundColor Gray
        foreach ($func in $notFound) {
            Write-Host "    [-] $func" -ForegroundColor Cyan
        }
    }

    # Check if LoadLibrary and GetProcAddress are imported (used for dynamic loading)
    Write-Host "`n[DYNAMIC LOADING INDICATORS]" -ForegroundColor Magenta
    $hasLoadLibrary = $false
    $hasGetProcAddress = $false

    foreach ($dll in $imports.Keys) {
        if ($imports[$dll] -match "LoadLibrary") {
            $hasLoadLibrary = $true
            Write-Host "  [+] LoadLibrary found - can load DLLs dynamically" -ForegroundColor Green
        }
        if ($imports[$dll] -match "GetProcAddress") {
            $hasGetProcAddress = $true
            Write-Host "  [+] GetProcAddress found - can resolve functions dynamically" -ForegroundColor Green
        }
    }

    if (-not $hasLoadLibrary) {
        Write-Host "  [-] LoadLibrary NOT found" -ForegroundColor Red
    }
    if (-not $hasGetProcAddress) {
        Write-Host "  [-] GetProcAddress NOT found" -ForegroundColor Red
    }

} elseif ($Detailed) {
    foreach ($dll in $imports.Keys | Sort-Object) {
        Write-Host "[$dll]" -ForegroundColor Yellow
        foreach ($func in $imports[$dll]) {
            Write-Host "  - $func" -ForegroundColor Gray
        }
        Write-Host ""
    }
} else {
    # Summary view
    $totalFunctions = 0
    foreach ($dll in $imports.Keys | Sort-Object) {
        $funcCount = $imports[$dll].Count
        $totalFunctions += $funcCount
        Write-Host "  [$dll] - $funcCount function(s)" -ForegroundColor Gray
    }
    Write-Host "`nTotal: $totalFunctions imported function(s)" -ForegroundColor White
    Write-Host "Use -Detailed to see all function names" -ForegroundColor DarkGray
    Write-Host "Use -SearchFunctions to check specific functions" -ForegroundColor DarkGray
    Write-Host ""
}
