#!/usr/bin/env pwsh
<#
.SYNOPSIS
    Builds and packages the Wildfire Buffet CLI with all dependencies
.DESCRIPTION
    This script:
    1. Builds the CLI as a single executable
    2. Copies Modules, ExecutionTypes, Packagers, Recipes, and Templates
    3. Creates a wildfire-config.json with relative paths
    4. Packages everything into a ZIP file
.PARAMETER OutputPath
    Where to create the package (default: ./dist)
.PARAMETER Configuration
    Build configuration (Debug or Release, default: Release)
#>

param(
    [string]$OutputPath = ".\dist",
    [string]$Configuration = "Release"
)

$ErrorActionPreference = "Stop"

# Get script directory and project root
$ScriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$ProjectRoot = Split-Path -Parent $ScriptDir

Write-Host "========================================" -ForegroundColor Cyan
Write-Host "Wildfire Buffet CLI Package Builder" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host ""

# Check for C++ build tools (required for Native AOT)
Write-Host "[Check] Verifying C++ build tools..." -ForegroundColor Yellow

$vswhere = "${env:ProgramFiles(x86)}\Microsoft Visual Studio\Installer\vswhere.exe"
$hasCppTools = $false

if (Test-Path $vswhere) {
    $vsPath = & $vswhere -latest -requires Microsoft.VisualStudio.Component.VC.Tools.x86.x64 -property installationPath
    if ($vsPath) {
        $hasCppTools = $true
        Write-Host "[OK] C++ build tools found: $vsPath" -ForegroundColor Green
    }
}

if (-not $hasCppTools) {
    Write-Host ""
    Write-Host "========================================" -ForegroundColor Red
    Write-Host "ERROR: C++ Build Tools Not Found" -ForegroundColor Red
    Write-Host "========================================" -ForegroundColor Red
    Write-Host ""
    Write-Host "Native AOT compilation requires Visual Studio C++ build tools." -ForegroundColor Yellow
    Write-Host ""
    Write-Host "Please install one of the following:" -ForegroundColor White
    Write-Host "  1. Visual Studio 2022 with 'Desktop development with C++' workload" -ForegroundColor Cyan
    Write-Host "  2. Visual Studio Build Tools 2022 with C++ tools" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "Download options:" -ForegroundColor White
    Write-Host "  Visual Studio 2022 Community (Free):" -ForegroundColor Gray
    Write-Host "    https://visualstudio.microsoft.com/downloads/" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "  Build Tools Only (Smaller download):" -ForegroundColor Gray
    Write-Host "    https://visualstudio.microsoft.com/downloads/#build-tools-for-visual-studio-2022" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "Installation steps:" -ForegroundColor White
    Write-Host "  1. Run the installer" -ForegroundColor Gray
    Write-Host "  2. Select 'Desktop development with C++' workload" -ForegroundColor Gray
    Write-Host "  3. Install (requires ~7GB disk space)" -ForegroundColor Gray
    Write-Host "  4. Restart your terminal and run this script again" -ForegroundColor Gray
    Write-Host ""

    $response = Read-Host "Would you like to open the download page now? (Y/N)"
    if ($response -eq 'Y' -or $response -eq 'y') {
        Start-Process "https://visualstudio.microsoft.com/downloads/#build-tools-for-visual-studio-2022"
    }

    exit 1
}

Write-Host ""

# Create output directory
$OutputPath = Join-Path $ProjectRoot $OutputPath
if (Test-Path $OutputPath) {
    Write-Host "[Info] Cleaning existing output directory..." -ForegroundColor Yellow
    Remove-Item -Path $OutputPath -Recurse -Force
}
New-Item -ItemType Directory -Path $OutputPath -Force | Out-Null

# Step 1: Publish CLI as Native AOT executable
Write-Host "[1/5] Publishing CLI as Native AOT executable..." -ForegroundColor Green
$PublishPath = Join-Path $ProjectRoot "WildfireBuffet.CLI\bin\$Configuration\net8.0\win-x64\publish"

Push-Location (Join-Path $ProjectRoot "WildfireBuffet.CLI")
try {
    dotnet publish -c $Configuration -r win-x64 -p:PublishAot=true | Out-Host
    if ($LASTEXITCODE -ne 0) {
        throw "Build failed with exit code $LASTEXITCODE"
    }
}
finally {
    Pop-Location
}

# Step 2: Create package directory structure
Write-Host "[2/5] Creating package structure..." -ForegroundColor Green
$PackageRoot = Join-Path $OutputPath "WildfireBuffet-CLI"
New-Item -ItemType Directory -Path $PackageRoot -Force | Out-Null

# Copy executable
Write-Host "  Copying executable..." -ForegroundColor Gray
Copy-Item -Path (Join-Path $PublishPath "WildfireBuffet.CLI.exe") -Destination $PackageRoot

# Step 3: Copy dependencies
Write-Host "[3/5] Copying dependencies..." -ForegroundColor Green

# Copy Modules directory
if (Test-Path (Join-Path $ProjectRoot "Modules")) {
    Write-Host "  Copying Modules..." -ForegroundColor Gray
    Copy-Item -Path (Join-Path $ProjectRoot "Modules") -Destination (Join-Path $PackageRoot "Modules") -Recurse
}

# Copy ExecutionTypes directory
if (Test-Path (Join-Path $ProjectRoot "ExecutionTypes")) {
    Write-Host "  Copying ExecutionTypes..." -ForegroundColor Gray
    Copy-Item -Path (Join-Path $ProjectRoot "ExecutionTypes") -Destination (Join-Path $PackageRoot "ExecutionTypes") -Recurse
}

# Copy Packagers directory
if (Test-Path (Join-Path $ProjectRoot "Packagers")) {
    Write-Host "  Copying Packagers..." -ForegroundColor Gray
    Copy-Item -Path (Join-Path $ProjectRoot "Packagers") -Destination (Join-Path $PackageRoot "Packagers") -Recurse
}

# Copy Recipes directory
if (Test-Path (Join-Path $ProjectRoot "Recipes")) {
    Write-Host "  Copying Recipes..." -ForegroundColor Gray
    Copy-Item -Path (Join-Path $ProjectRoot "Recipes") -Destination (Join-Path $PackageRoot "Recipes") -Recurse
}

# Copy Templates directory
if (Test-Path (Join-Path $ProjectRoot "Templates")) {
    Write-Host "  Copying Templates..." -ForegroundColor Gray
    Copy-Item -Path (Join-Path $ProjectRoot "Templates") -Destination (Join-Path $PackageRoot "Templates") -Recurse
}

# Create output directory
New-Item -ItemType Directory -Path (Join-Path $PackageRoot "output") -Force | Out-Null

# Step 4: Create wildfire-config.json with relative paths
Write-Host "[4/5] Creating configuration file..." -ForegroundColor Green
$ConfigContent = @{
    modulesDirectory = "Modules"
    executionTypesDirectory = "ExecutionTypes"
    packagersDirectory = "Packagers"
    recipesDirectory = "Recipes"
    templatesDirectory = "Templates"
    outputDirectory = "output"
} | ConvertTo-Json -Depth 10

$ConfigPath = Join-Path $PackageRoot "wildfire-config.json"
$ConfigContent | Out-File -FilePath $ConfigPath -Encoding UTF8

Write-Host "  Created: wildfire-config.json" -ForegroundColor Gray

# Create README
$ReadmeContent = @"
# Wildfire Buffet CLI - Portable Package

This is a portable package of the Wildfire Buffet CLI tool.

## Contents
- WildfireBuffet.CLI.exe - Main executable (single-file, self-contained)
- Modules/ - Action and evasion modules
- ExecutionTypes/ - Code generation templates
- Packagers/ - Output packaging options (ZIP, ISO)
- Recipes/ - Preconfigured payload recipes
- Templates/ - Code templates
- output/ - Generated payloads will be saved here
- wildfire-config.json - Configuration file

## Usage

### Interactive Mode
Simply run the executable:
``````
WildfireBuffet.CLI.exe
``````

### Command-Line Mode (with recipe)
``````
WildfireBuffet.CLI.exe --config Recipes\your-recipe.json
``````

## Configuration
Edit wildfire-config.json to customize directory paths.
All paths are relative to the executable location.

## Author
ril3y

## Note
This tool is for authorized security testing, defensive security, CTF challenges,
and educational contexts only. Use responsibly.
"@

$ReadmePath = Join-Path $PackageRoot "README.txt"
$ReadmeContent | Out-File -FilePath $ReadmePath -Encoding UTF8

# Step 5: Create ZIP package
Write-Host "[5/5] Creating ZIP package..." -ForegroundColor Green
$ZipFileName = "WildfireBuffet-CLI-win-x64.zip"
$ZipPath = Join-Path $OutputPath $ZipFileName

# Use .NET compression
Add-Type -AssemblyName System.IO.Compression.FileSystem
[System.IO.Compression.ZipFile]::CreateFromDirectory($PackageRoot, $ZipPath)

# Get package size
$PackageSize = (Get-Item $ZipPath).Length / 1MB

Write-Host ""
Write-Host "========================================" -ForegroundColor Green
Write-Host "Package created successfully!" -ForegroundColor Green
Write-Host "========================================" -ForegroundColor Green
Write-Host "Location: $ZipPath" -ForegroundColor Cyan
Write-Host "Size: $([math]::Round($PackageSize, 2)) MB" -ForegroundColor Cyan
Write-Host ""
Write-Host "To use:" -ForegroundColor Yellow
Write-Host "  1. Extract the ZIP file" -ForegroundColor White
Write-Host "  2. Run WildfireBuffet.CLI.exe" -ForegroundColor White
Write-Host ""

# Show package contents
Write-Host "Package contents:" -ForegroundColor Yellow
Get-ChildItem -Path $PackageRoot -Recurse -File |
    Select-Object @{Name='Path';Expression={$_.FullName.Replace($PackageRoot + '\', '')}},
                  @{Name='Size';Expression={"{0:N0} KB" -f ($_.Length / 1KB)}} |
    Format-Table -AutoSize
