/*
 * MODULE: Dynamic Import Resolver
 * ID: dynamic-import-resolver
 * DESCRIPTION: Loads multiple DLLs and resolves API functions at runtime using LoadLibrary/GetProcAddress, then calls them via function pointers
 * CATEGORY: Import Obfuscation
 * SUPPORTED_EXECUTION_TYPES: native-executable,native-executable-aot
 *
 * PARAMETERS:
 * @DllImports|string|kernel32.dll:GetComputerNameW,GetLocalTime,GetModuleFileNameW;advapi32.dll:GetUserNameW|||required|DLL-to-API mappings in format "dll1:api1,api2;dll2:api3,api4"
 */

// USINGS
using System;
using System.Runtime.InteropServices;

// CODE
// Dynamic import resolution with function pointer table storage - mimics real malware behavior
var dllImports = @DllImports;

Console.WriteLine("[*] Dynamic Import Resolver - Function Pointer Table\n");

// Parse format: "dll1:api1,api2;dll2:api3,api4"
var dllMappings = dllImports.Split(new[] { ';' }, StringSplitOptions.RemoveEmptyEntries);

// Count total APIs to allocate function pointer table
int totalApis = 0;
foreach (var mapping in dllMappings)
{
    var parts = mapping.Split(new[] { ':' }, 2);
    if (parts.Length == 2)
    {
        var apiList = parts[1].Split(new[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
        totalApis += apiList.Length;
    }
}

// Allocate function pointer table in memory (like malware does)
int pointerSize = IntPtr.Size;
IntPtr functionTable = Marshal.AllocHGlobal(totalApis * pointerSize);
Console.WriteLine($"[*] Function Pointer Table allocated at: 0x{functionTable.ToInt64():X}");
Console.WriteLine($"[*] Table size: {totalApis} pointers ({totalApis * pointerSize} bytes)\n");

// Resolve APIs and store pointers in the table
int tableIndex = 0;
foreach (var mapping in dllMappings)
{
    var parts = mapping.Split(new[] { ':' }, 2);
    if (parts.Length != 2)
    {
        Console.WriteLine($"[-] Invalid mapping format: {mapping}");
        continue;
    }

    var dllName = parts[0].Trim();
    var apiList = parts[1].Split(new[] { ',' }, StringSplitOptions.RemoveEmptyEntries);

    var hModule = NativeMethods.LoadLibrary(dllName);

    if (hModule == IntPtr.Zero)
    {
        Console.WriteLine($"[-] Failed to load {dllName}");
        continue;
    }

    foreach (var apiName in apiList)
    {
        var trimmedApi = apiName.Trim();
        var procAddr = NativeMethods.GetProcAddress(hModule, trimmedApi);

        if (procAddr == IntPtr.Zero)
        {
            Console.WriteLine($"[-] {trimmedApi} -> FAILED");
        }
        else
        {
            // Calculate where in our table this pointer will be stored
            IntPtr tableSlot = functionTable + (tableIndex * pointerSize);

            // Store the function pointer in our table
            Marshal.WriteIntPtr(tableSlot, procAddr);

            // Display: API name -> actual function address (stored at table address)
            Console.WriteLine($"[+] {trimmedApi} -> 0x{procAddr.ToInt64():X} (table[{tableIndex}] @ 0x{tableSlot.ToInt64():X})");

            tableIndex++;
        }
    }
}

Console.WriteLine($"\n[*] Resolved {tableIndex} function pointers");
Console.WriteLine($"[*] Function table ready for use at: 0x{functionTable.ToInt64():X}");

// Demonstrate using stored pointers from the table
Console.WriteLine("\n[*] Demonstrating function calls using stored pointers...\n");

// Example: Read back pointer from table and invoke it
if (tableIndex > 0)
{
    // Get the first function pointer from our table
    IntPtr firstFuncPtr = Marshal.ReadIntPtr(functionTable);
    Console.WriteLine($"[*] Example: Reading pointer from table[0]: 0x{firstFuncPtr.ToInt64():X}");
}

// Clean up (in real malware, table would persist for the lifetime of the process)
// Marshal.FreeHGlobal(functionTable);

// Static class to hold P/Invoke declarations (must be at end for top-level statements)
static class NativeMethods
{
    [DllImport("kernel32.dll", CharSet = CharSet.Auto, SetLastError = true)]
    public static extern IntPtr LoadLibrary(string lpFileName);

    [DllImport("kernel32.dll", CharSet = CharSet.Ansi, SetLastError = true)]
    public static extern IntPtr GetProcAddress(IntPtr hModule, string lpProcName);
}
