/*
 * MODULE: System Info Writer
 * ID: system-info-writer
 * DESCRIPTION: Writes detailed system information to a file for reconnaissance
 * CATEGORY: Information Gathering
 * SUPPORTED_EXECUTION_TYPES: native-executable,native-executable-aot,powershell-script
 *
 * PARAMETERS:
 * @OutputPath|string|C:\Temp\sysinfo.txt||||Output file path for system information
 * @IncludeEnvironment|bool|true||||Include environment variables
 * @IncludeProcesses|bool|false||||Include running process list
 */

// USINGS
using System;
using System.IO;
using System.Diagnostics;
using System.Linq;

// CODE
try
{
    Console.WriteLine("[+] System Info Writer - Starting...");

    var outputPath = @OutputPath;
    var lines = new System.Collections.Generic.List<string>();

    // Basic system information
    lines.Add("=== SYSTEM INFORMATION ===");
    lines.Add($"Computer Name: {Environment.MachineName}");
    lines.Add($"User Name: {Environment.UserName}");
    lines.Add($"OS Version: {Environment.OSVersion}");
    lines.Add($"64-bit OS: {Environment.Is64BitOperatingSystem}");
    lines.Add($"64-bit Process: {Environment.Is64BitProcess}");
    lines.Add($"Processor Count: {Environment.ProcessorCount}");
    lines.Add($"System Directory: {Environment.SystemDirectory}");
    lines.Add($"Current Directory: {Environment.CurrentDirectory}");
    lines.Add($"CLR Version: {Environment.Version}");
    lines.Add("");

    // Environment variables (if enabled)
    if (@IncludeEnvironment)
    {
        lines.Add("=== ENVIRONMENT VARIABLES ===");
        foreach (System.Collections.DictionaryEntry env in Environment.GetEnvironmentVariables())
        {
            lines.Add($"{env.Key}={env.Value}");
        }
        lines.Add("");
    }

    // Running processes (if enabled)
    if (@IncludeProcesses)
    {
        lines.Add("=== RUNNING PROCESSES ===");
        var processes = Process.GetProcesses()
            .OrderBy(p => p.ProcessName)
            .Take(50); // Limit to first 50 processes

        foreach (var proc in processes)
        {
            try
            {
                lines.Add($"{proc.ProcessName} (PID: {proc.Id})");
            }
            catch
            {
                // Skip processes we can't access
            }
        }
        lines.Add("");
    }

    // Write to file
    File.WriteAllLines(outputPath, lines);
    Console.WriteLine($"[+] System information written to: {outputPath}");
    Console.WriteLine($"[+] Total lines written: {lines.Count}");
}
catch (Exception ex)
{
    Console.WriteLine($"[-] Error writing system info: {ex.Message}");
}
