/*
 * MODULE: Registry Operations (Native)
 * ID: registry-ops-native
 * DESCRIPTION: Registry operations using .NET Registry API for native executables
 * CATEGORY: System Actions
 * SUPPORTED_EXECUTION_TYPES: native-executable,native-executable-aot
 *
 * PARAMETERS:
 * @Operation|choice|Read|Read:Write:Delete:Test:CreateKey:DeleteKey:Enumerate||required|Registry operation to perform
 * @Path|string|HKCU:\Software\Microsoft\Windows\CurrentVersion|||placeholder=HKCU:\Software\TestApp,required|Registry path (PowerShell format: HKCU:\, HKLM:\, HKCR:\, HKU:\, HKCC:\)
 * @Name|string|TestValue|||placeholder=AppVersion,optional|Registry value name (leave empty for default value)
 * @Value|string||||placeholder=C:\Program Files,optional|Value to write (for Write operation)
 * @Type|choice|String|String:DWord:QWord:Binary:MultiString:ExpandString||required|Registry value type
 * @Recursive|bool|false||||Recursive operation for Enumerate/Delete operations
 * @Force|bool|false||||Force operation even if it might fail (creates intermediate keys, overwrites existing values)
 * @ShowValues|bool|true||||Show value data when enumerating registry keys
 */

// USINGS
using System;
using Microsoft.Win32;
using System.Linq;
using System.Collections.Generic;

// CODE
// Registry Operations (Native) Module
// Provides comprehensive registry access using .NET Registry API

var operation = @Operation;
var registryPath = @Path;
var valueName = @Name;
var valueData = @Value;
var valueType = @Type;
var recursive = @Recursive;
var force = @Force;
var showValues = @ShowValues;

Console.WriteLine($"[Registry] ========================================");
Console.WriteLine($"[Registry] Registry Operations (Native)");
Console.WriteLine($"[Registry] ========================================");
Console.WriteLine($"[Registry] Operation: {operation}");
Console.WriteLine($"[Registry] Path: {registryPath}");
if (!string.IsNullOrEmpty(valueName))
{
    Console.WriteLine($"[Registry] Value Name: {valueName}");
}
Console.WriteLine();

try
{
    // Parse registry path (PowerShell format: HKCU:\Software\...)
    RegistryKey baseKey;
    string subKeyPath;

    if (registryPath.StartsWith("HKCU:\\", StringComparison.OrdinalIgnoreCase) ||
        registryPath.StartsWith("HKEY_CURRENT_USER\\", StringComparison.OrdinalIgnoreCase))
    {
        baseKey = Registry.CurrentUser;
        subKeyPath = registryPath.StartsWith("HKCU:\\", StringComparison.OrdinalIgnoreCase)
            ? registryPath.Substring(6)
            : registryPath.Substring(18);
    }
    else if (registryPath.StartsWith("HKLM:\\", StringComparison.OrdinalIgnoreCase) ||
             registryPath.StartsWith("HKEY_LOCAL_MACHINE\\", StringComparison.OrdinalIgnoreCase))
    {
        baseKey = Registry.LocalMachine;
        subKeyPath = registryPath.StartsWith("HKLM:\\", StringComparison.OrdinalIgnoreCase)
            ? registryPath.Substring(6)
            : registryPath.Substring(19);
    }
    else if (registryPath.StartsWith("HKCR:\\", StringComparison.OrdinalIgnoreCase) ||
             registryPath.StartsWith("HKEY_CLASSES_ROOT\\", StringComparison.OrdinalIgnoreCase))
    {
        baseKey = Registry.ClassesRoot;
        subKeyPath = registryPath.StartsWith("HKCR:\\", StringComparison.OrdinalIgnoreCase)
            ? registryPath.Substring(6)
            : registryPath.Substring(18);
    }
    else if (registryPath.StartsWith("HKU:\\", StringComparison.OrdinalIgnoreCase) ||
             registryPath.StartsWith("HKEY_USERS\\", StringComparison.OrdinalIgnoreCase))
    {
        baseKey = Registry.Users;
        subKeyPath = registryPath.StartsWith("HKU:\\", StringComparison.OrdinalIgnoreCase)
            ? registryPath.Substring(5)
            : registryPath.Substring(12);
    }
    else if (registryPath.StartsWith("HKCC:\\", StringComparison.OrdinalIgnoreCase) ||
             registryPath.StartsWith("HKEY_CURRENT_CONFIG\\", StringComparison.OrdinalIgnoreCase))
    {
        baseKey = Registry.CurrentConfig;
        subKeyPath = registryPath.StartsWith("HKCC:\\", StringComparison.OrdinalIgnoreCase)
            ? registryPath.Substring(6)
            : registryPath.Substring(20);
    }
    else
    {
        // Default to HKCU if no recognized prefix
        Console.WriteLine("[Registry] WARNING: No recognized hive prefix, defaulting to HKCU");
        baseKey = Registry.CurrentUser;
        subKeyPath = registryPath;
    }

    // Execute operation
    switch (operation.ToLowerInvariant())
    {
        case "read":
            {
                using var key = baseKey.OpenSubKey(subKeyPath, false);
                if (key == null)
                {
                    Console.WriteLine($"[Registry] ERROR: Registry key does not exist");
                    Console.WriteLine($"[Registry] Path: {registryPath}");
                    Environment.Exit(1);
                }

                if (string.IsNullOrEmpty(valueName))
                {
                    // Read default value
                    var defaultValue = key.GetValue("");
                    Console.WriteLine($"[Registry] Default value: {defaultValue ?? "(not set)"}");
                }
                else
                {
                    // Read named value
                    var value = key.GetValue(valueName);
                    if (value == null)
                    {
                        Console.WriteLine($"[Registry] ERROR: Registry value does not exist: {valueName}");
                        Environment.Exit(1);
                    }

                    var kind = key.GetValueKind(valueName);
                    Console.WriteLine($"[Registry] Value Type: {kind}");
                    Console.WriteLine($"[Registry] Value Data: {value}");

                    // Display additional info for certain types
                    if (kind == RegistryValueKind.MultiString && value is string[] multiString)
                    {
                        Console.WriteLine($"[Registry] Multi-string values:");
                        for (int i = 0; i < multiString.Length; i++)
                        {
                            Console.WriteLine($"[Registry]   [{i}] {multiString[i]}");
                        }
                    }
                    else if (kind == RegistryValueKind.Binary && value is byte[] binary)
                    {
                        Console.WriteLine($"[Registry] Binary data length: {binary.Length} bytes");
                        Console.WriteLine($"[Registry] Binary data (hex): {BitConverter.ToString(binary.Take(16).ToArray())}...");
                    }
                }

                Console.WriteLine($"[Registry] Operation completed successfully");
                break;
            }

        case "write":
            {
                // Open or create key
                using var key = force
                    ? baseKey.CreateSubKey(subKeyPath, true)
                    : baseKey.OpenSubKey(subKeyPath, true);

                if (key == null)
                {
                    Console.WriteLine($"[Registry] ERROR: Cannot open registry key for writing");
                    Console.WriteLine($"[Registry] Path: {registryPath}");
                    Console.WriteLine($"[Registry] Hint: Use Force=true to create intermediate keys");
                    Environment.Exit(1);
                }

                // Convert value type
                RegistryValueKind regValueKind;
                object valueToWrite;

                switch (valueType.ToLowerInvariant())
                {
                    case "string":
                        regValueKind = RegistryValueKind.String;
                        valueToWrite = valueData;
                        break;

                    case "dword":
                        regValueKind = RegistryValueKind.DWord;
                        if (!int.TryParse(valueData, out int dwordValue))
                        {
                            Console.WriteLine($"[Registry] ERROR: Invalid DWord value: {valueData}");
                            Environment.Exit(1);
                        }
                        valueToWrite = dwordValue;
                        break;

                    case "qword":
                        regValueKind = RegistryValueKind.QWord;
                        if (!long.TryParse(valueData, out long qwordValue))
                        {
                            Console.WriteLine($"[Registry] ERROR: Invalid QWord value: {valueData}");
                            Environment.Exit(1);
                        }
                        valueToWrite = qwordValue;
                        break;

                    case "binary":
                        regValueKind = RegistryValueKind.Binary;
                        // Parse hex string (e.g., "01,02,03,04")
                        try
                        {
                            var hexValues = valueData.Split(new[] { ',', ' ' }, StringSplitOptions.RemoveEmptyEntries);
                            valueToWrite = hexValues.Select(h => Convert.ToByte(h, 16)).ToArray();
                        }
                        catch
                        {
                            Console.WriteLine($"[Registry] ERROR: Invalid binary data format. Use hex comma-separated (e.g., '01,02,03')");
                            Environment.Exit(1);
                            return;
                        }
                        break;

                    case "multistring":
                        regValueKind = RegistryValueKind.MultiString;
                        valueToWrite = valueData.Split(new[] { '|' }, StringSplitOptions.None);
                        break;

                    case "expandstring":
                        regValueKind = RegistryValueKind.ExpandString;
                        valueToWrite = valueData;
                        break;

                    default:
                        Console.WriteLine($"[Registry] ERROR: Unknown registry value type: {valueType}");
                        Environment.Exit(1);
                        return;
                }

                // Write value
                key.SetValue(valueName ?? "", valueToWrite, regValueKind);
                Console.WriteLine($"[Registry] SUCCESS: Registry value written");
                Console.WriteLine($"[Registry] Value: {valueName ?? "(default)"}");
                Console.WriteLine($"[Registry] Type: {regValueKind}");
                Console.WriteLine($"[Registry] Data: {valueData}");
                break;
            }

        case "delete":
            {
                using var key = baseKey.OpenSubKey(subKeyPath, true);
                if (key == null)
                {
                    Console.WriteLine($"[Registry] WARNING: Registry key does not exist, nothing to delete");
                    break;
                }

                if (string.IsNullOrEmpty(valueName))
                {
                    Console.WriteLine($"[Registry] ERROR: Cannot delete registry key from value operation");
                    Console.WriteLine($"[Registry] Hint: Use 'DeleteKey' operation to delete keys");
                    Environment.Exit(1);
                }

                // Delete value
                try
                {
                    key.DeleteValue(valueName, !force);
                    Console.WriteLine($"[Registry] SUCCESS: Registry value deleted");
                    Console.WriteLine($"[Registry] Value: {valueName}");
                }
                catch (ArgumentException)
                {
                    Console.WriteLine($"[Registry] WARNING: Registry value does not exist: {valueName}");
                    if (!force)
                    {
                        Environment.Exit(1);
                    }
                }
                break;
            }

        case "test":
            {
                using var key = baseKey.OpenSubKey(subKeyPath, false);
                if (key == null)
                {
                    Console.WriteLine($"[Registry] Result: Key does NOT exist");
                    Console.WriteLine($"[Registry] Path: {registryPath}");
                    break;
                }

                if (string.IsNullOrEmpty(valueName))
                {
                    Console.WriteLine($"[Registry] Result: Key EXISTS");
                }
                else
                {
                    var value = key.GetValue(valueName);
                    if (value == null)
                    {
                        Console.WriteLine($"[Registry] Result: Key exists but value does NOT exist");
                        Console.WriteLine($"[Registry] Value: {valueName}");
                    }
                    else
                    {
                        Console.WriteLine($"[Registry] Result: Key and value EXIST");
                        Console.WriteLine($"[Registry] Value: {valueName} = {value}");
                    }
                }
                break;
            }

        case "createkey":
            {
                using var key = baseKey.CreateSubKey(subKeyPath, true);
                if (key == null)
                {
                    Console.WriteLine($"[Registry] ERROR: Failed to create registry key");
                    Environment.Exit(1);
                }

                Console.WriteLine($"[Registry] SUCCESS: Registry key created (or already exists)");
                Console.WriteLine($"[Registry] Path: {registryPath}");
                break;
            }

        case "deletekey":
            {
                try
                {
                    if (recursive)
                    {
                        baseKey.DeleteSubKeyTree(subKeyPath, !force);
                        Console.WriteLine($"[Registry] SUCCESS: Registry key and subkeys deleted recursively");
                    }
                    else
                    {
                        baseKey.DeleteSubKey(subKeyPath, !force);
                        Console.WriteLine($"[Registry] SUCCESS: Registry key deleted");
                    }
                    Console.WriteLine($"[Registry] Path: {registryPath}");
                }
                catch (ArgumentException)
                {
                    Console.WriteLine($"[Registry] WARNING: Registry key does not exist");
                    if (!force)
                    {
                        Environment.Exit(1);
                    }
                }
                break;
            }

        case "enumerate":
            {
                using var key = baseKey.OpenSubKey(subKeyPath, false);
                if (key == null)
                {
                    Console.WriteLine($"[Registry] ERROR: Registry key does not exist");
                    Environment.Exit(1);
                }

                Console.WriteLine($"[Registry] Enumerating: {registryPath}");
                Console.WriteLine();

                // Enumerate subkeys
                var subKeyNames = key.GetSubKeyNames();
                Console.WriteLine($"[Registry] Subkeys ({subKeyNames.Length}):");
                foreach (var subKeyName in subKeyNames)
                {
                    Console.WriteLine($"[Registry]   {subKeyName}\\");

                    // Recursive enumeration
                    if (recursive)
                    {
                        try
                        {
                            using var subKey = key.OpenSubKey(subKeyName, false);
                            if (subKey != null)
                            {
                                var subSubKeys = subKey.GetSubKeyNames();
                                foreach (var ssk in subSubKeys)
                                {
                                    Console.WriteLine($"[Registry]     {ssk}\\");
                                }
                            }
                        }
                        catch
                        {
                            Console.WriteLine($"[Registry]     (access denied)");
                        }
                    }
                }

                Console.WriteLine();

                // Enumerate values
                var valueNames = key.GetValueNames();
                Console.WriteLine($"[Registry] Values ({valueNames.Length}):");
                foreach (var name in valueNames)
                {
                    try
                    {
                        var value = key.GetValue(name);
                        var kind = key.GetValueKind(name);

                        var displayName = string.IsNullOrEmpty(name) ? "(default)" : name;

                        if (showValues)
                        {
                            var valuePreview = value?.ToString() ?? "(not set)";
                            if (valuePreview.Length > 50)
                            {
                                valuePreview = valuePreview.Substring(0, 50) + "...";
                            }
                            Console.WriteLine($"[Registry]   {displayName} ({kind}) = {valuePreview}");
                        }
                        else
                        {
                            Console.WriteLine($"[Registry]   {displayName} ({kind})");
                        }
                    }
                    catch
                    {
                        Console.WriteLine($"[Registry]   {name} (error reading)");
                    }
                }

                Console.WriteLine();
                Console.WriteLine($"[Registry] Enumeration completed");
                break;
            }

        default:
            Console.WriteLine($"[Registry] ERROR: Unknown operation: {operation}");
            Console.WriteLine($"[Registry] Valid operations: Read, Write, Delete, Test, CreateKey, DeleteKey, Enumerate");
            Environment.Exit(1);
            break;
    }

    Console.WriteLine($"[Registry] ========================================");
}
catch (UnauthorizedAccessException ex)
{
    Console.WriteLine($"[Registry] ERROR: Access denied");
    Console.WriteLine($"[Registry] Details: {ex.Message}");
    Console.WriteLine($"[Registry] Hint: Administrator privileges may be required for this operation");
    Environment.Exit(1);
}
catch (System.Security.SecurityException ex)
{
    Console.WriteLine($"[Registry] ERROR: Security exception");
    Console.WriteLine($"[Registry] Details: {ex.Message}");
    Console.WriteLine($"[Registry] Hint: Insufficient permissions for registry access");
    Environment.Exit(1);
}
catch (Exception ex)
{
    Console.WriteLine($"[Registry] ERROR: Registry operation failed");
    Console.WriteLine($"[Registry] Exception: {ex.GetType().Name}");
    Console.WriteLine($"[Registry] Message: {ex.Message}");
    Console.WriteLine($"[Registry] ========================================");
    Environment.Exit(1);
}
