/*
 * MODULE: Registry Operations
 * ID: registry-ops
 * DESCRIPTION: Advanced registry operations with validation and conditional logic for real-world testing
 * CATEGORY: System Actions
 * SUPPORTED_EXECUTION_TYPES: native-executable,native-executable-aot
 *
 * PARAMETERS:
 * @Operation|choice|ReadAndValidate|Read:ReadAndValidate:Create:CreateIfMissing:Delete:CheckExists||required|Operation type
 * @KeyPath|string|HKEY_CURRENT_USER\\Software\\Microsoft\\Windows\\CurrentVersion|||placeholder=HKCU\\Software\\TestApp,required|Registry key path (HKLM, HKCU, HKCR prefixes supported)
 * @ValueName|string|TestValue|||placeholder=AppVersion,required|Registry value name to read/write
 * @ValueData|string||||optional|Expected value for validation, or value to write for Create operations
 * @FailOnMismatch|bool|true||||Exit with error if validation fails
 * @CreateIfMissing|bool|false||||Create key/value if missing (for ReadAndValidate operation)
 */

// USINGS
using System;
using Microsoft.Win32;

// CODE
try
{
    var operation = @Operation;
    var keyPath = @KeyPath;
    var valueName = @ValueName;
    var valueData = @ValueData;
    var failOnMismatch = @FailOnMismatch;
    var createIfMissing = @CreateIfMissing;

    Console.WriteLine($"[Registry] Operation: {operation}");
    Console.WriteLine($"[Registry] Target: {keyPath}\\{valueName}");

    // Parse registry hive from path
    RegistryKey baseKey;
    string subKeyPath;

    if (keyPath.StartsWith("HKLM\\", StringComparison.OrdinalIgnoreCase) ||
        keyPath.StartsWith("HKEY_LOCAL_MACHINE\\", StringComparison.OrdinalIgnoreCase))
    {
        baseKey = RegistryKey.OpenBaseKey(RegistryHive.LocalMachine, RegistryView.Registry64);
        subKeyPath = keyPath.StartsWith("HKLM\\", StringComparison.OrdinalIgnoreCase)
            ? keyPath.Substring(5)
            : keyPath.Substring(19);
    }
    else if (keyPath.StartsWith("HKCU\\", StringComparison.OrdinalIgnoreCase) ||
             keyPath.StartsWith("HKEY_CURRENT_USER\\", StringComparison.OrdinalIgnoreCase))
    {
        baseKey = RegistryKey.OpenBaseKey(RegistryHive.CurrentUser, RegistryView.Registry64);
        subKeyPath = keyPath.StartsWith("HKCU\\", StringComparison.OrdinalIgnoreCase)
            ? keyPath.Substring(5)
            : keyPath.Substring(18);
    }
    else if (keyPath.StartsWith("HKCR\\", StringComparison.OrdinalIgnoreCase) ||
             keyPath.StartsWith("HKEY_CLASSES_ROOT\\", StringComparison.OrdinalIgnoreCase))
    {
        baseKey = RegistryKey.OpenBaseKey(RegistryHive.ClassesRoot, RegistryView.Registry64);
        subKeyPath = keyPath.StartsWith("HKCR\\", StringComparison.OrdinalIgnoreCase)
            ? keyPath.Substring(5)
            : keyPath.Substring(18);
    }
    else
    {
        // Default to HKLM if no prefix specified
        baseKey = RegistryKey.OpenBaseKey(RegistryHive.LocalMachine, RegistryView.Registry64);
        subKeyPath = keyPath;
        Console.WriteLine("[Registry] No hive prefix detected, defaulting to HKLM");
    }

    // Handle each operation type
    switch (operation.ToLowerInvariant())
    {
        case "read":
            {
                using var key = baseKey.OpenSubKey(subKeyPath, false);
                if (key == null)
                {
                    Console.WriteLine($"[Registry] ERROR: Key does not exist: {subKeyPath}");
                    Environment.Exit(1);
                }

                var value = key.GetValue(valueName);
                if (value == null)
                {
                    Console.WriteLine($"[Registry] ERROR: Value does not exist: {valueName}");
                    Environment.Exit(1);
                }

                var valueKind = key.GetValueKind(valueName);
                Console.WriteLine($"[Registry] Read successful");
                Console.WriteLine($"[Registry] Type: {valueKind}");
                Console.WriteLine($"[Registry] Value: {value}");
                break;
            }

        case "readandvalidate":
            {
                using var key = baseKey.OpenSubKey(subKeyPath, false);

                if (key == null)
                {
                    if (createIfMissing)
                    {
                        Console.WriteLine($"[Registry] Key does not exist, creating: {subKeyPath}");
                        using var newKey = baseKey.CreateSubKey(subKeyPath, true);
                        newKey?.SetValue(valueName, valueData, RegistryValueKind.String);
                        Console.WriteLine($"[Registry] Created: {valueName} = {valueData}");
                        break;
                    }
                    else
                    {
                        Console.WriteLine($"[Registry] ERROR: Key does not exist: {subKeyPath}");
                        if (failOnMismatch)
                            Environment.Exit(1);
                        break;
                    }
                }

                var value = key.GetValue(valueName);
                if (value == null)
                {
                    if (createIfMissing)
                    {
                        Console.WriteLine($"[Registry] Value does not exist, creating: {valueName}");
                        key.Close();
                        using var writeKey = baseKey.OpenSubKey(subKeyPath, true);
                        writeKey?.SetValue(valueName, valueData, RegistryValueKind.String);
                        Console.WriteLine($"[Registry] Created: {valueName} = {valueData}");
                        break;
                    }
                    else
                    {
                        Console.WriteLine($"[Registry] ERROR: Value does not exist: {valueName}");
                        if (failOnMismatch)
                            Environment.Exit(1);
                        break;
                    }
                }

                var actualValue = value.ToString();
                Console.WriteLine($"[Registry] Current value: {actualValue}");
                Console.WriteLine($"[Registry] Expected value: {valueData}");

                if (string.IsNullOrEmpty(valueData))
                {
                    Console.WriteLine($"[Registry] Validation skipped (no expected value provided)");
                }
                else if (actualValue.Equals(valueData, StringComparison.OrdinalIgnoreCase))
                {
                    Console.WriteLine($"[Registry] Validation PASSED: Values match");
                }
                else
                {
                    Console.WriteLine($"[Registry] Validation FAILED: Values do not match");
                    if (failOnMismatch)
                    {
                        Console.WriteLine($"[Registry] Exiting with error code due to FailOnMismatch=true");
                        Environment.Exit(1);
                    }
                }
                break;
            }

        case "create":
            {
                using var key = baseKey.CreateSubKey(subKeyPath, true);
                if (key == null)
                {
                    Console.WriteLine($"[Registry] ERROR: Failed to create/open key: {subKeyPath}");
                    Environment.Exit(1);
                }

                // Detect value type - if it's a number, use DWORD, otherwise String
                if (int.TryParse(valueData, out int intValue))
                {
                    key.SetValue(valueName, intValue, RegistryValueKind.DWord);
                    Console.WriteLine($"[Registry] Created/Updated (DWORD): {valueName} = {intValue}");
                }
                else
                {
                    key.SetValue(valueName, valueData, RegistryValueKind.String);
                    Console.WriteLine($"[Registry] Created/Updated (String): {valueName} = {valueData}");
                }
                break;
            }

        case "createifmissing":
            {
                using var checkKey = baseKey.OpenSubKey(subKeyPath, false);

                if (checkKey == null)
                {
                    Console.WriteLine($"[Registry] Key does not exist, creating: {subKeyPath}");
                    using var newKey = baseKey.CreateSubKey(subKeyPath, true);
                    if (int.TryParse(valueData, out int intValue))
                    {
                        newKey?.SetValue(valueName, intValue, RegistryValueKind.DWord);
                        Console.WriteLine($"[Registry] Created (DWORD): {valueName} = {intValue}");
                    }
                    else
                    {
                        newKey?.SetValue(valueName, valueData, RegistryValueKind.String);
                        Console.WriteLine($"[Registry] Created (String): {valueName} = {valueData}");
                    }
                }
                else
                {
                    var existingValue = checkKey.GetValue(valueName);
                    if (existingValue == null)
                    {
                        Console.WriteLine($"[Registry] Value does not exist, creating: {valueName}");
                        checkKey.Close();
                        using var writeKey = baseKey.OpenSubKey(subKeyPath, true);
                        if (int.TryParse(valueData, out int intValue))
                        {
                            writeKey?.SetValue(valueName, intValue, RegistryValueKind.DWord);
                            Console.WriteLine($"[Registry] Created (DWORD): {valueName} = {intValue}");
                        }
                        else
                        {
                            writeKey?.SetValue(valueName, valueData, RegistryValueKind.String);
                            Console.WriteLine($"[Registry] Created (String): {valueName} = {valueData}");
                        }
                    }
                    else
                    {
                        Console.WriteLine($"[Registry] Value already exists: {valueName} = {existingValue}");
                        Console.WriteLine($"[Registry] Skipped creation (already exists)");
                    }
                }
                break;
            }

        case "delete":
            {
                using var key = baseKey.OpenSubKey(subKeyPath, true);
                if (key == null)
                {
                    Console.WriteLine($"[Registry] Key does not exist: {subKeyPath}");
                    Console.WriteLine($"[Registry] Nothing to delete");
                    break;
                }

                if (string.IsNullOrEmpty(valueName))
                {
                    // Delete the entire key
                    key.Close();
                    baseKey.DeleteSubKeyTree(subKeyPath, false);
                    Console.WriteLine($"[Registry] Deleted key: {subKeyPath}");
                }
                else
                {
                    // Delete just the value
                    var value = key.GetValue(valueName);
                    if (value == null)
                    {
                        Console.WriteLine($"[Registry] Value does not exist: {valueName}");
                        Console.WriteLine($"[Registry] Nothing to delete");
                    }
                    else
                    {
                        key.DeleteValue(valueName, false);
                        Console.WriteLine($"[Registry] Deleted value: {valueName}");
                    }
                }
                break;
            }

        case "checkexists":
            {
                using var key = baseKey.OpenSubKey(subKeyPath, false);

                if (key == null)
                {
                    Console.WriteLine($"[Registry] Key does NOT exist: {subKeyPath}");
                    if (failOnMismatch)
                    {
                        Console.WriteLine($"[Registry] Exiting with error code due to FailOnMismatch=true");
                        Environment.Exit(1);
                    }
                    break;
                }

                if (string.IsNullOrEmpty(valueName))
                {
                    Console.WriteLine($"[Registry] Key EXISTS: {subKeyPath}");
                }
                else
                {
                    var value = key.GetValue(valueName);
                    if (value == null)
                    {
                        Console.WriteLine($"[Registry] Key exists but value does NOT exist: {valueName}");
                        if (failOnMismatch)
                        {
                            Console.WriteLine($"[Registry] Exiting with error code due to FailOnMismatch=true");
                            Environment.Exit(1);
                        }
                    }
                    else
                    {
                        Console.WriteLine($"[Registry] Key and value EXIST: {valueName} = {value}");
                    }
                }
                break;
            }

        default:
            Console.WriteLine($"[Registry] ERROR: Unknown operation: {operation}");
            Console.WriteLine($"[Registry] Valid operations: Read, ReadAndValidate, Create, CreateIfMissing, Delete, CheckExists");
            Environment.Exit(1);
            break;
    }

    baseKey.Close();
    Console.WriteLine($"[Registry] Operation completed successfully");
}
catch (UnauthorizedAccessException ex)
{
    Console.WriteLine($"[Registry] ERROR: Access denied. Administrator privileges may be required.");
    Console.WriteLine($"[Registry] Details: {ex.Message}");
    Environment.Exit(1);
}
catch (System.Security.SecurityException ex)
{
    Console.WriteLine($"[Registry] ERROR: Security exception. Insufficient permissions.");
    Console.WriteLine($"[Registry] Details: {ex.Message}");
    Environment.Exit(1);
}
catch (Exception ex)
{
    Console.WriteLine($"[Registry] ERROR: Registry operation failed");
    Console.WriteLine($"[Registry] Exception: {ex.GetType().Name}");
    Console.WriteLine($"[Registry] Message: {ex.Message}");
    Environment.Exit(1);
}
