/*
 * MODULE: PowerShell Registry Manipulation
 * ID: ps-registry-manipulation
 * DESCRIPTION: Registry operations using PowerShell registry cmdlets and .NET Registry API
 * CATEGORY: System Actions
 * SUPPORTED_EXECUTION_TYPES: powershell-script,native-executable,native-executable-aot
 *
 * PARAMETERS:
 * @Operation|choice|Write|Read:Write:Delete:Test:CreateKey:DeleteKey:Enumerate||required|Registry operation to perform
 * @Path|string|HKCU:\Software\Microsoft\Windows\CurrentVersion\Run||placeholder=Registry\Path|Registry key path (PowerShell format - HKCU colon backslash = HKEY_CURRENT_USER)
 * @Name|string|TestApp||placeholder=ValueName|Registry value name (e.g., TestApp creates an entry in the Run key)
 * @Value|string|C:\Windows\System32\notepad.exe||placeholder=C:\Path\To\App.exe|Value data (e.g., path to executable for Run key)
 * @Type|choice|String|String:DWord:QWord:Binary:MultiString:ExpandString||required|Registry value type
 * @Recursive|bool|false||||Recursive operation for Enumerate/Delete operations
 * @Force|bool|true||||Force operation even if it might fail (creates intermediate keys, overwrites existing values)
 * @ShowValues|bool|true||||Show value data when enumerating registry keys
 */

// USINGS
// No C# usings needed - this outputs PowerShell script code

// CODE
# PowerShell Registry Manipulation
# Registry operations using PowerShell cmdlets

$operation = "@Operation"
$registryPath = "@Path"
$valueName = "@Name"
$valueData = "@Value"
$valueType = "@Type"
$recursive = @Recursive
$force = @Force
$showValues = @ShowValues

Write-Host "[Registry] ========================================"
Write-Host "[Registry] PowerShell Registry Manipulation"
Write-Host "[Registry] ========================================"
Write-Host "[Registry] Operation: $operation"
Write-Host "[Registry] Path: $registryPath"
if ($valueName) {
    Write-Host "[Registry] Value Name: $valueName"
}
Write-Host ""

try {
    switch ($operation) {
        "Read" {
            if ($valueName) {
                $item = Get-ItemProperty -Path $registryPath -Name $valueName -ErrorAction Stop
                $valueKind = (Get-Item -Path $registryPath).GetValueKind($valueName)
                Write-Host "[Registry] Value: $($item.$valueName)"
                Write-Host "[Registry] Type: $valueKind"
            } else {
                $item = Get-ItemProperty -Path $registryPath -ErrorAction Stop
                Write-Host "[Registry] Default value: $($item.'(default)')"
            }
            Write-Host "[Registry] Operation completed successfully"
        }
        "Write" {
            $setParams = @{
                Path = $registryPath
                Name = $valueName
                Value = $valueData
            }

            if ($force) {
                $setParams.Force = $true
            }

            switch ($valueType) {
                "String" { $setParams.Type = "String" }
                "DWord" {
                    $setParams.Type = "DWord"
                    $setParams.Value = [int]$valueData
                }
                "QWord" {
                    $setParams.Type = "QWord"
                    $setParams.Value = [long]$valueData
                }
                "Binary" {
                    $setParams.Type = "Binary"
                    $hexValues = $valueData -split '[,\s]+'
                    $setParams.Value = $hexValues | ForEach-Object { [byte]("0x$_") }
                }
                "MultiString" {
                    $setParams.Type = "MultiString"
                    $setParams.Value = $valueData -split '\|'
                }
                "ExpandString" { $setParams.Type = "ExpandString" }
            }

            Set-ItemProperty @setParams -ErrorAction Stop
            Write-Host "[Registry] SUCCESS: Registry value written"
            Write-Host "[Registry] Value: $($valueName)"
            Write-Host "[Registry] Type: $valueType"
            Write-Host "[Registry] Data: $valueData"
        }
        "Delete" {
            if ([string]::IsNullOrEmpty($valueName)) {
                Write-Host "[Registry] ERROR: Cannot delete registry key from value operation"
                Write-Host "[Registry] Hint: Use 'DeleteKey' operation to delete keys"
                exit 1
            }
            Remove-ItemProperty -Path $registryPath -Name $valueName -ErrorAction Stop
            Write-Host "[Registry] SUCCESS: Registry value deleted"
            Write-Host "[Registry] Value: $valueName"
        }
        "Test" {
            $exists = Test-Path -Path $registryPath
            if ($exists -and $valueName) {
                $item = Get-ItemProperty -Path $registryPath -ErrorAction SilentlyContinue
                $valueExists = $null -ne $item.$valueName
                if ($valueExists) {
                    Write-Host "[Registry] Result: Key and value EXIST"
                    Write-Host "[Registry] Value: $valueName = $($item.$valueName)"
                } else {
                    Write-Host "[Registry] Result: Key exists but value does NOT exist"
                    Write-Host "[Registry] Value: $valueName"
                }
            } elseif ($exists) {
                Write-Host "[Registry] Result: Key EXISTS"
            } else {
                Write-Host "[Registry] Result: Key does NOT exist"
                Write-Host "[Registry] Path: $registryPath"
            }
        }
        "CreateKey" {
            if (!(Test-Path $registryPath)) {
                New-Item -Path $registryPath -Force:$force -ErrorAction Stop | Out-Null
                Write-Host "[Registry] SUCCESS: Registry key created"
            } else {
                Write-Host "[Registry] SUCCESS: Registry key already exists"
            }
            Write-Host "[Registry] Path: $registryPath"
        }
        "DeleteKey" {
            if (Test-Path $registryPath) {
                Remove-Item -Path $registryPath -Recurse:$recursive -Force:$force -ErrorAction Stop
                Write-Host "[Registry] SUCCESS: Registry key deleted"
                Write-Host "[Registry] Path: $registryPath"
            } else {
                Write-Host "[Registry] WARNING: Registry key does not exist"
                if (!$force) {
                    exit 1
                }
            }
        }
        "Enumerate" {
            if (!(Test-Path $registryPath)) {
                Write-Host "[Registry] ERROR: Registry key does not exist"
                exit 1
            }

            Write-Host "[Registry] Enumerating: $registryPath"
            Write-Host ""

            # Enumerate subkeys
            $subKeys = Get-ChildItem -Path $registryPath -Recurse:$recursive -ErrorAction SilentlyContinue
            Write-Host "[Registry] Subkeys ($($subKeys.Count)):"
            foreach ($subKey in $subKeys) {
                Write-Host "[Registry]   $($subKey.PSChildName)\"
            }

            Write-Host ""

            # Enumerate values
            $item = Get-Item -Path $registryPath
            $valueNames = $item.GetValueNames()
            Write-Host "[Registry] Values ($($valueNames.Count)):"
            foreach ($name in $valueNames) {
                try {
                    $value = $item.GetValue($name)
                    $kind = $item.GetValueKind($name)
                    $displayName = if ([string]::IsNullOrEmpty($name)) { "(default)" } else { $name }

                    if ($showValues) {
                        $valuePreview = $value.ToString()
                        if ($valuePreview.Length -gt 50) {
                            $valuePreview = $valuePreview.Substring(0, 50) + "..."
                        }
                        Write-Host "[Registry]   $displayName ($kind) = $valuePreview"
                    } else {
                        Write-Host "[Registry]   $displayName ($kind)"
                    }
                }
                catch {
                    Write-Host "[Registry]   $name (error reading)"
                }
            }

            Write-Host ""
            Write-Host "[Registry] Enumeration completed"
        }
        default {
            Write-Host "[Registry] ERROR: Unknown operation: $operation"
            Write-Host "[Registry] Valid operations: Read, Write, Delete, Test, CreateKey, DeleteKey, Enumerate"
            exit 1
        }
    }
    Write-Host "[Registry] ========================================"
}
catch {
    Write-Host "[Registry] ERROR: Registry operation failed" -ForegroundColor Red
    Write-Host "[Registry] Exception: $($_.Exception.GetType().Name)" -ForegroundColor Red
    Write-Host "[Registry] Message: $($_.Exception.Message)" -ForegroundColor Red
    Write-Host "[Registry] ========================================" -ForegroundColor Red
    exit 1
}
