/*
 * MODULE: PowerShell Web Request
 * ID: ps-web-request
 * DESCRIPTION: HTTP/HTTPS web requests using PowerShell Invoke-WebRequest cmdlet
 * CATEGORY: Network Actions
 * SUPPORTED_EXECUTION_TYPES: powershell-script,native-executable,native-executable-aot
 *
 * PARAMETERS:
 * @Urls|stringlist|https://api.github.com/zen|||placeholder=https://api.example.com/data,required|URLs to request (comma-separated list)
 */

// USINGS
using System;
using System.Net.Http;
using System.Threading.Tasks;
using System.Collections.Generic;
using System.Linq;

// CODE
// PowerShell Web Request Module
// Simple HTTP/HTTPS requests using HttpClient

var urls = @Urls.Split(',', StringSplitOptions.RemoveEmptyEntries).Select(u => u.Trim()).ToArray();

Console.WriteLine($"[WebRequest] Starting requests for {urls.Length} URL(s)");
Console.WriteLine();

// Configure HTTP client with sensible defaults
var handler = new HttpClientHandler
{
    AllowAutoRedirect = true,
    MaxAutomaticRedirections = 10
};

using var httpClient = new HttpClient(handler)
{
    Timeout = TimeSpan.FromSeconds(30)
};

// Set a standard user agent
httpClient.DefaultRequestHeaders.Add("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36");

// Execute requests for each URL
int successCount = 0;
int failureCount = 0;

foreach (var url in urls)
{
    try
    {
        // Ensure URL has protocol
        var fullUrl = url;
        if (!fullUrl.StartsWith("http://", StringComparison.OrdinalIgnoreCase) &&
            !fullUrl.StartsWith("https://", StringComparison.OrdinalIgnoreCase))
        {
            fullUrl = "https://" + fullUrl;
        }

        Console.WriteLine($"[WebRequest] Requesting: {fullUrl}");

        var startTime = DateTime.UtcNow;
        var response = httpClient.GetAsync(fullUrl).GetAwaiter().GetResult();
        var elapsed = DateTime.UtcNow - startTime;

        using (response)
        {
            Console.WriteLine($"[WebRequest] Status: {(int)response.StatusCode} {response.StatusCode}");
            Console.WriteLine($"[WebRequest] Response time: {elapsed.TotalMilliseconds:F0} ms");
            Console.WriteLine($"[WebRequest] Content-Type: {response.Content.Headers.ContentType?.MediaType ?? "unknown"}");

            if (response.Content.Headers.ContentLength.HasValue)
            {
                Console.WriteLine($"[WebRequest] Content-Length: {response.Content.Headers.ContentLength.Value} bytes");
            }

            // Check if request was successful
            if (response.IsSuccessStatusCode)
            {
                Console.WriteLine($"[WebRequest] SUCCESS: Request completed successfully");
                successCount++;
            }
            else
            {
                Console.WriteLine($"[WebRequest] WARNING: Request returned non-success status code");
                failureCount++;
            }
        }

        Console.WriteLine();
    }
    catch (TaskCanceledException ex)
    {
        Console.WriteLine($"[WebRequest] ERROR: Request timed out after 30 seconds");
        Console.WriteLine($"[WebRequest] URL: {url}");
        Console.WriteLine($"[WebRequest] Details: {ex.Message}");
        failureCount++;
        Console.WriteLine();
    }
    catch (HttpRequestException ex)
    {
        Console.WriteLine($"[WebRequest] ERROR: HTTP request failed");
        Console.WriteLine($"[WebRequest] URL: {url}");
        Console.WriteLine($"[WebRequest] Details: {ex.Message}");
        failureCount++;
        Console.WriteLine();
    }
    catch (Exception ex)
    {
        Console.WriteLine($"[WebRequest] ERROR: Unexpected error occurred");
        Console.WriteLine($"[WebRequest] URL: {url}");
        Console.WriteLine($"[WebRequest] Exception: {ex.GetType().Name}");
        Console.WriteLine($"[WebRequest] Message: {ex.Message}");
        failureCount++;
        Console.WriteLine();
    }
}

// Summary
Console.WriteLine($"[WebRequest] ========================================");
Console.WriteLine($"[WebRequest] Summary:");
Console.WriteLine($"[WebRequest]   Total URLs: {urls.Length}");
Console.WriteLine($"[WebRequest]   Successful: {successCount}");
Console.WriteLine($"[WebRequest]   Failed: {failureCount}");
Console.WriteLine($"[WebRequest] ========================================");

// Exit with error code if any requests failed
if (failureCount > 0)
{
    Console.WriteLine($"[WebRequest] WARNING: {failureCount} request(s) failed");
}
