/*
 * MODULE: LNK URL Launcher
 * ID: lnk-url-launcher
 * DESCRIPTION: Creates LNK shortcuts that open specified URLs in the default or specified browser
 * CATEGORY: LNK Actions
 * SUPPORTED_EXECUTION_TYPES: lnk-shortcut
 *
 * PARAMETERS:
 * @Urls|stringlist|https://portal.office.com|||placeholder=https://portal.office.com,required|Comma-separated list of URLs to open
 * @Browser|choice|default|chrome:firefox:edge:default||required|Browser to use for opening URLs
 *
 * USAGE:
 * This module configures LNK shortcuts to open URLs when clicked.
 * - For single URL: Creates shortcut that opens that URL
 * - For multiple URLs: Opens all URLs sequentially
 * - Browser selection: Can specify which browser to use or use system default
 *
 * BROWSER PATHS:
 * - default: Uses system default handler (no browser specified)
 * - chrome: C:\Program Files\Google\Chrome\Application\chrome.exe
 * - firefox: C:\Program Files\Mozilla Firefox\firefox.exe
 * - edge: C:\Program Files (x86)\Microsoft\Edge\Application\msedge.exe
 */

// USINGS
using System;
using System.Diagnostics;
using System.Runtime.InteropServices;

// CODE
Console.WriteLine("[*] LNK URL Launcher Module");
Console.WriteLine($"[*] URLs: {@Urls}");
Console.WriteLine($"[*] Browser: {@Browser}");

try
{
    var urls = @Urls.Split(',');
    var browser = @Browser.ToLower();

    foreach (var url in urls)
    {
        var cleanUrl = url.Trim();
        if (string.IsNullOrEmpty(cleanUrl))
            continue;

        // Ensure URL has protocol
        if (!cleanUrl.StartsWith("http://", StringComparison.OrdinalIgnoreCase) &&
            !cleanUrl.StartsWith("https://", StringComparison.OrdinalIgnoreCase))
        {
            cleanUrl = "https://" + cleanUrl;
        }

        Console.WriteLine($"[*] Opening URL: {cleanUrl}");

        // Determine browser path
        string browserPath = browser switch
        {
            "chrome" => @"C:\Program Files\Google\Chrome\Application\chrome.exe",
            "firefox" => @"C:\Program Files\Mozilla Firefox\firefox.exe",
            "edge" => @"C:\Program Files (x86)\Microsoft\Edge\Application\msedge.exe",
            _ => "" // default - use system handler
        };

        // Launch URL
        if (string.IsNullOrEmpty(browserPath))
        {
            // Use default browser via ShellExecute
            Process.Start(new ProcessStartInfo
            {
                FileName = cleanUrl,
                UseShellExecute = true
            });
        }
        else
        {
            // Use specific browser
            if (System.IO.File.Exists(browserPath))
            {
                Process.Start(browserPath, cleanUrl);
            }
            else
            {
                Console.WriteLine($"[!] Browser not found at: {browserPath}, using default");
                Process.Start(new ProcessStartInfo
                {
                    FileName = cleanUrl,
                    UseShellExecute = true
                });
            }
        }

        Console.WriteLine($"[+] URL opened: {cleanUrl}");

        // Small delay between URLs to prevent overwhelming the browser
        if (urls.Length > 1)
        {
            System.Threading.Thread.Sleep(500);
        }
    }

    Console.WriteLine("[+] All URLs processed successfully");
}
catch (Exception ex)
{
    Console.WriteLine($"[-] Error launching URLs: {ex.Message}");
}
