/*
 * MODULE: Named Pipe Server with Mutex
 * ID: named-pipe-server
 * DESCRIPTION: Creates a named pipe server and mutex for IPC demonstration - common malware indicators
 * CATEGORY: Inter-Process Communication
 * SUPPORTED_EXECUTION_TYPES: native-executable,native-executable-aot,powershell-script
 *
 * PARAMETERS:
 * @PipeName|string|TestPipe|||required|Name of the pipe (without \\.\pipe\ prefix)
 * @MutexName|string|Global\TestMutex|||required|Name of the mutex (can include Global\ or Local\ prefix)
 * @WaitTimeSeconds|int|5|1|60||Time to keep pipe open waiting for connections (seconds)
 * @MaxConnections|int|1|1|10||Maximum number of client connections to accept
 * @ShowStatus|bool|true|||Show detailed status messages
 */

// USINGS
using System;
using System.IO;
using System.IO.Pipes;
using System.Threading;
using System.Threading.Tasks;
using System.Text;

// CODE
try
{
    var pipeName = @PipeName;
    var mutexName = @MutexName;
    var waitTimeSeconds = @WaitTimeSeconds;
    var maxConnections = @MaxConnections;
    var showStatus = @ShowStatus;

    if (showStatus)
    {
        Console.WriteLine("[*] Named Pipe & Mutex Demo - Starting...");
        Console.WriteLine($"[*] Pipe Name: \\\\.\\pipe\\{pipeName}");
        Console.WriteLine($"[*] Mutex Name: {mutexName}");
        Console.WriteLine();
    }

    // Create a named mutex (common malware indicator)
    bool createdNew;
    Mutex mutex = null;

    try
    {
        mutex = new Mutex(true, mutexName, out createdNew);

        if (createdNew)
        {
            if (showStatus)
            {
                Console.WriteLine($"[+] Successfully created mutex: {mutexName}");
            }
        }
        else
        {
            if (showStatus)
            {
                Console.WriteLine($"[!] Mutex already exists: {mutexName}");
                Console.WriteLine("[!] Another instance may be running");
            }
        }
    }
    catch (Exception ex)
    {
        Console.WriteLine($"[-] Failed to create mutex: {ex.Message}");
    }

    // Create a named pipe server (common C2 communication method)
    if (showStatus)
    {
        Console.WriteLine($"[*] Creating named pipe server...");
    }

    using (var pipeServer = new NamedPipeServerStream(
        pipeName,
        PipeDirection.InOut,
        maxConnections,
        PipeTransmissionMode.Byte,
        PipeOptions.Asynchronous))
    {
        if (showStatus)
        {
            Console.WriteLine($"[+] Named pipe created successfully");
            Console.WriteLine($"[*] Waiting for client connections (timeout: {waitTimeSeconds}s)...");
        }

        // Wait for connection with timeout
        var connectTask = pipeServer.WaitForConnectionAsync();
        var timeoutTask = Task.Delay(TimeSpan.FromSeconds(waitTimeSeconds));

        var completedTask = Task.WhenAny(connectTask, timeoutTask).Result;

        if (completedTask == connectTask)
        {
            // Client connected
            if (showStatus)
            {
                Console.WriteLine("[+] Client connected!");
            }

            // Read message from client
            try
            {
                var buffer = new byte[1024];
                int bytesRead = pipeServer.Read(buffer, 0, buffer.Length);

                if (bytesRead > 0)
                {
                    string message = Encoding.UTF8.GetString(buffer, 0, bytesRead);
                    if (showStatus)
                    {
                        Console.WriteLine($"[+] Received: {message}");
                    }

                    // Send response
                    var response = Encoding.UTF8.GetBytes("Message received");
                    pipeServer.Write(response, 0, response.Length);
                    pipeServer.Flush();

                    if (showStatus)
                    {
                        Console.WriteLine("[+] Sent response to client");
                    }
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine($"[-] Error during communication: {ex.Message}");
            }
        }
        else
        {
            // Timeout reached
            if (showStatus)
            {
                Console.WriteLine($"[*] Timeout reached - no client connected in {waitTimeSeconds} seconds");
            }
        }
    }

    if (showStatus)
    {
        Console.WriteLine("[*] Pipe server closed");
    }

    // Clean up mutex
    if (mutex != null)
    {
        mutex.ReleaseMutex();
        mutex.Dispose();

        if (showStatus)
        {
            Console.WriteLine($"[+] Released mutex: {mutexName}");
        }
    }

    if (showStatus)
    {
        Console.WriteLine("[+] Demo completed successfully");
    }
}
catch (Exception ex)
{
    Console.WriteLine($"[-] Error: {ex.Message}");
    Console.WriteLine($"[-] Stack trace: {ex.StackTrace}");
}
