/*
 * EXECUTION TYPE: Native Executable (Self-Contained)
 * ID: native-executable
 * DESCRIPTION: Compiles to .exe with .NET runtime included (~33MB, fast build, no dependencies)
 * CATEGORY: Executable
 * FILE EXTENSION: .exe
 *
 * PARAMETERS:
 * (none)
 *
 * SUPPORTED MODULES: * (all modules)
 *
 * HOW TO USE:
 * This execution type compiles your C# modules to a Windows executable with the .NET runtime bundled.
 * - File size: ~33MB (includes full runtime)
 * - Build time: Fast (2-5 seconds)
 * - Requirements: None
 * - Best for: Quick testing, compatibility
 */

using System;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Collections.Generic;
using WildfireBuffet.Core.Abstractions;
using WildfireBuffet.Core.Engine;
using WildfireBuffet.Core.Helpers;

public class NativeExecutable : ExecutionTypeBase
{
    private readonly CompilationEngine _compiler;

    public NativeExecutable()
    {
        _compiler = new CompilationEngine();
    }

    public override ExecutionTypeDescriptor Descriptor => new()
    {
        Id = "native-executable",
        Name = "Native Executable (Self-Contained)",
        Description = "Windows executable (.exe) with .NET runtime bundled (~33MB, fast build)",
        FileExtension = ".exe",
        Icon = "⚙️",
        Category = "Executable",
        SupportedModules = new[] { "*" }, // Supports all modules
        ConfigurableParameters = new Dictionary<string, string>()
    };

    public override async Task<ExecutionResult> GenerateAsync(ExecutionContext context)
    {
        // Generate source code from modules
        var sourceCode = GenerateExecutableCode(context);

        // Compile to executable (self-contained with .NET runtime)
        var tempPath = Path.Combine(Path.GetTempPath(), $"goblin_native_{Guid.NewGuid():N}.exe");

        try
        {
            var compilationResult = await _compiler.CompileToExeAsync(sourceCode, tempPath, useNativeAot: false);

            if (!compilationResult.Success)
            {
                var errors = string.Join("\n", compilationResult.Diagnostics);
                return ExecutionResult.Failed($"Compilation failed:\n{errors}");
            }

            // Read the compiled executable
            var exeBytes = await File.ReadAllBytesAsync(tempPath);

            // Clean up temp file
            try { File.Delete(tempPath); } catch { }

            return ExecutionResult.Succeeded(exeBytes);
        }
        catch (Exception ex)
        {
            return ExecutionResult.Failed($"Exception during compilation: {ex.Message}");
        }
    }

    /// <summary>
    /// Extracts static class declarations from module code and returns them separately.
    /// Static classes must be placed after top-level statements in C# 10+.
    /// </summary>

    /// <summary>
    /// Generate complete C# source code for standalone executable
    /// </summary>
    private string GenerateExecutableCode(ExecutionContext context)
    {
        var code = new System.Text.StringBuilder();

        // Add usings - start with core System usings
        var coreUsings = new HashSet<string> { "System", "System.Linq", "System.Collections.Generic" };

        var allUsings = context.Modules
            .SelectMany(m => m.Usings)
            .Concat(coreUsings)
            .Distinct()
            .OrderBy(u => u);

        foreach (var usingNamespace in allUsings)
        {
            code.AppendLine($"using {usingNamespace};");
        }

        code.AppendLine();
        code.AppendLine("// Generated by Goblin King");
        code.AppendLine($"// Sample: {context.SampleNumber}");
        code.AppendLine($"// Timestamp: {DateTime.UtcNow:yyyy-MM-dd HH:mm:ss} UTC");
        code.AppendLine();

        // Main execution
        code.AppendLine("// Main execution");
        code.AppendLine("try");
        code.AppendLine("{");

        // Collect static classes from all modules
        var allStaticClasses = new List<string>();

        // Execute each module in order
        foreach (var module in context.Modules)
        {
            code.AppendLine($"    // === {module.Name} ===");
            var moduleCode = module.GetSubstitutedCode();

            // Extract static classes (they must go after top-level statements)
            var (mainCode, classDeclarations) = CSharpHelper.ExtractStaticClasses(moduleCode);
            allStaticClasses.AddRange(classDeclarations);

            // Indent main module code
            var lines = mainCode.Split('\n');
            foreach (var line in lines)
            {
                if (!string.IsNullOrWhiteSpace(line))
                {
                    code.AppendLine($"    {line.TrimEnd()}");
                }
            }
            code.AppendLine();
        }

        code.AppendLine("}");
        code.AppendLine("catch (Exception ex)");
        code.AppendLine("{");
        code.AppendLine("    Console.WriteLine($\"[ERROR] {ex.Message}\");");
        code.AppendLine("    Environment.Exit(1);");
        code.AppendLine("}");
        code.AppendLine();

        // Add pause at end
        code.AppendLine("Console.WriteLine(\"\\nExecution completed.\");");
        code.AppendLine("Console.WriteLine(\"Press any key to exit...\");");
        code.AppendLine("Console.ReadKey(true);");

        // Add static classes after all top-level statements
        if (allStaticClasses.Any())
        {
            code.AppendLine();
            code.AppendLine("// Static classes from modules (must be after top-level statements)");
            foreach (var staticClass in allStaticClasses)
            {
                code.AppendLine(staticClass);
                code.AppendLine();
            }
        }

        return code.ToString();
    }
}
