/*
 * EXECUTION TYPE: Windows Shortcut (LNK)
 * ID: lnk-shortcut
 * DESCRIPTION: Creates Windows .lnk shortcut files with polymorphic output support
 * CATEGORY: Shortcut
 * FILE EXTENSION: .lnk
 *
 * PARAMETERS:
 * @TargetType:choice:powershell,cmd,url:powershell::What the shortcut executes (PowerShell, CMD, or URL)
 * @WindowStyle:choice:Normal,Minimized,Maximized,Hidden:Normal::Window display style
 * @IconPath:string::::Path to icon file (optional)
 * @IconIndex:int:0:::Icon index within icon file
 * @HotKey:string::::Keyboard shortcut (e.g., CTRL+ALT+X)
 * @Description:string:Windows Shortcut:::Tooltip description
 *
 * SUPPORTED MODULES:
 * * (all modules marked with lnk-shortcut execution type)
 *
 * POLYMORPHIC FEATURES:
 * - Randomized icon index based on sample number
 * - Varied hotkey combinations
 * - Different window styles
 * - Random description/comment text
 * - Randomized working directory paths
 *
 * HOW IT WORKS:
 * Creates .lnk shortcut files that can execute PowerShell commands, CMD commands, or open URLs.
 * Each generated shortcut has different metadata to ensure unique hashes even with identical recipes.
 */

using System;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using WildfireBuffet.Core.Abstractions;
using WildfireBuffet.Core.Helpers;

public class LnkShortcut : ExecutionTypeBase
{
    public override ExecutionTypeDescriptor Descriptor => new()
    {
        Id = "lnk-shortcut",
        Name = "Windows Shortcut (LNK)",
        Description = "Windows .lnk shortcut file that can execute commands, PowerShell, or launch URLs",
        FileExtension = ".lnk",
        Icon = "🔗",
        Category = "Shortcut",
        SupportedModules = new[] { "*" }, // Support all modules marked with lnk-shortcut
        ConfigurableParameters = new Dictionary<string, string>
        {
            { "TargetType", "choice - powershell|cmd|url (what the shortcut executes)" },
            { "WindowStyle", "choice - Normal|Minimized|Maximized|Hidden" },
            { "IconPath", "string - Path to icon file (optional)" },
            { "IconIndex", "int - Icon index in icon file" },
            { "HotKey", "string - Keyboard shortcut (e.g., CTRL+ALT+X)" },
            { "Description", "string - Tooltip description" }
        }
    };

    public override async Task<ExecutionResult> GenerateAsync(ExecutionContext context)
    {
        // Validate module compatibility
        var validationError = ValidateModuleCompatibility(context);
        if (validationError != null)
        {
            return validationError;
        }

        // Get configuration with polymorphic defaults
        var targetType = GetConfig(context, "TargetType", "powershell");
        var windowStyle = GetConfig(context, "WindowStyle", "Normal");
        var iconPath = GetConfig(context, "IconPath", "");
        var iconIndex = GetConfig(context, "IconIndex", GetPolymorphicIconIndex(context.SampleNumber));
        var hotKey = GetConfig(context, "HotKey", GetPolymorphicHotKey(context.SampleNumber));
        var description = GetConfig(context, "Description", GetPolymorphicDescription(context.SampleNumber));

        try
        {
            // Generate target command from modules
            var targetCommand = GenerateTargetCommand(context, targetType);
            var arguments = "";

            // Determine actual target executable based on target type
            string targetPath;
            switch (targetType?.ToLower())
            {
                case "powershell":
                    targetPath = "C:\\Windows\\System32\\WindowsPowerShell\\v1.0\\powershell.exe";
                    arguments = $"-ExecutionPolicy Bypass -NoProfile -WindowStyle Hidden -Command \"{targetCommand}\"";
                    break;

                case "cmd":
                    targetPath = "C:\\Windows\\System32\\cmd.exe";
                    arguments = $"/c {targetCommand}";
                    break;

                case "url":
                    // For URLs, target is the URL itself
                    targetPath = targetCommand;
                    arguments = "";
                    break;

                default:
                    return ExecutionResult.Failed($"Unknown TargetType: {targetType}");
            }

            // Generate the .lnk file
            var lnkBytes = await Task.Run(() => CreateLnkFile(
                targetPath,
                arguments,
                windowStyle,
                iconPath,
                iconIndex,
                hotKey,
                description,
                GetPolymorphicWorkingDirectory(context.SampleNumber)
            ));

            var result = ExecutionResult.Succeeded(lnkBytes);
            result.Diagnostics.Add($"Generated LNK shortcut for {targetType}");
            result.Diagnostics.Add($"Target: {targetPath}");
            result.Diagnostics.Add($"Arguments: {arguments}");
            result.Diagnostics.Add($"Window Style: {windowStyle}");
            result.Diagnostics.Add($"Description: {description}");
            result.Diagnostics.Add($"HotKey: {hotKey}");
            result.Diagnostics.Add($"Icon Index: {iconIndex} (polymorphic)");
            result.Diagnostics.Add($"Modules processed: {context.Modules.Count}");

            return result;
        }
        catch (Exception ex)
        {
            return ExecutionResult.Failed($"LNK generation failed: {ex.Message}\n{ex.StackTrace}");
        }
    }

    /// <summary>
    /// Generate the target command from configured modules
    /// </summary>
    private string GenerateTargetCommand(ExecutionContext context, string targetType)
    {
        var command = new StringBuilder();

        // Check if we have URL-based modules
        var urlModule = context.Modules.FirstOrDefault(m => m.Id == "lnk-url-launcher");
        if (urlModule != null)
        {
            // For URL launcher, just return the URL
            if (urlModule.ConfiguredValues.TryGetValue("Urls", out var urlsObj) && urlsObj is string urls)
            {
                var urlList = urls.Split(',').Select(u => u.Trim()).ToArray();
                if (urlList.Length > 0)
                {
                    return urlList[0]; // Return first URL
                }
            }
        }

        // Check for command executor module
        var cmdModule = context.Modules.FirstOrDefault(m => m.Id == "lnk-command-executor");
        if (cmdModule != null)
        {
            if (cmdModule.ConfiguredValues.TryGetValue("Command", out var cmdObj) && cmdObj is string cmd)
            {
                var args = "";
                if (cmdModule.ConfiguredValues.TryGetValue("Arguments", out var argsObj) && argsObj is string argsStr)
                {
                    args = argsStr;
                }
                return string.IsNullOrEmpty(args) ? cmd : $"{cmd} {args}";
            }
        }

        // If no specific module, generate from all module code
        foreach (var module in context.Modules)
        {
            var moduleCode = module.GetSubstitutedCode();

            // Clean up the code for command line execution
            var cleanCode = moduleCode
                .Replace("\r\n", "; ")
                .Replace("\n", "; ")
                .Replace("\"", "`\"");

            command.Append(cleanCode);
            command.Append("; ");
        }

        return command.ToString().TrimEnd(';', ' ');
    }

    /// <summary>
    /// Create a Windows .lnk file using COM automation with late-binding
    /// </summary>
    private byte[] CreateLnkFile(
        string targetPath,
        string arguments,
        string windowStyle,
        string iconPath,
        int iconIndex,
        string hotKey,
        string description,
        string workingDirectory)
    {
        var tempLnkPath = Path.Combine(Path.GetTempPath(), $"shortcut_{Guid.NewGuid():N}.lnk");

        try
        {
            // Create WScript.Shell COM object using late-binding
            Type shellType = Type.GetTypeFromProgID("WScript.Shell");
            if (shellType == null)
            {
                throw new InvalidOperationException("WScript.Shell COM object not available");
            }

            dynamic shell = Activator.CreateInstance(shellType);
            dynamic shortcut = shell.CreateShortcut(tempLnkPath);

            // Set shortcut properties
            shortcut.TargetPath = targetPath;
            shortcut.Arguments = arguments;
            shortcut.WorkingDirectory = workingDirectory;
            shortcut.Description = description;

            // Set window style
            int windowStyleValue = windowStyle?.ToLower() switch
            {
                "minimized" => 7,
                "maximized" => 3,
                "hidden" => 7,
                _ => 1 // Normal
            };
            shortcut.WindowStyle = windowStyleValue;

            // Set icon if provided
            if (!string.IsNullOrEmpty(iconPath) && File.Exists(iconPath))
            {
                shortcut.IconLocation = $"{iconPath},{iconIndex}";
            }

            // Set hotkey if provided
            if (!string.IsNullOrEmpty(hotKey))
            {
                shortcut.Hotkey = hotKey;
            }

            // Save the shortcut
            shortcut.Save();

            // Release COM objects
            Marshal.ReleaseComObject(shortcut);
            Marshal.ReleaseComObject(shell);

            // Read the generated .lnk file
            var lnkBytes = File.ReadAllBytes(tempLnkPath);

            // Clean up temp file
            try { File.Delete(tempLnkPath); } catch { }

            return lnkBytes;
        }
        catch (Exception ex)
        {
            // Clean up temp file if it exists
            try { if (File.Exists(tempLnkPath)) File.Delete(tempLnkPath); } catch { }
            throw new InvalidOperationException($"Failed to create LNK file: {ex.Message}", ex);
        }
    }

    /// <summary>
    /// Generate polymorphic icon index based on sample number
    /// </summary>
    private int GetPolymorphicIconIndex(int sampleNumber)
    {
        // Cycle through icon indices 0-9
        return sampleNumber % 10;
    }

    /// <summary>
    /// Generate polymorphic hotkey based on sample number
    /// </summary>
    private string GetPolymorphicHotKey(int sampleNumber)
    {
        var hotkeys = new[]
        {
            "",                     // No hotkey
            "CTRL+ALT+D",
            "CTRL+ALT+E",
            "CTRL+ALT+F",
            "CTRL+SHIFT+G",
            "CTRL+SHIFT+H",
            "ALT+SHIFT+J",
            "CTRL+ALT+K",
            "CTRL+SHIFT+L",
            "CTRL+ALT+M"
        };

        return hotkeys[sampleNumber % hotkeys.Length];
    }

    /// <summary>
    /// Generate polymorphic description based on sample number
    /// </summary>
    private string GetPolymorphicDescription(int sampleNumber)
    {
        var descriptions = new[]
        {
            "Windows Shortcut",
            "Application Launcher",
            "Quick Access Link",
            "System Utility",
            "Program Shortcut",
            "Desktop Link",
            "File Opener",
            "Utility Launcher",
            "Application Link",
            "System Shortcut"
        };

        var randomSuffix = new[]
        {
            "",
            " - Click to run",
            " - Double-click to execute",
            " - Launch application",
            " - Open file",
            " - Run program",
            " - Execute utility",
            " - Start application",
            " - Quick access",
            " - System tool"
        };

        var baseDesc = descriptions[sampleNumber % descriptions.Length];
        var suffix = randomSuffix[(sampleNumber / descriptions.Length) % randomSuffix.Length];

        return baseDesc + suffix;
    }

    /// <summary>
    /// Generate polymorphic working directory based on sample number
    /// </summary>
    private string GetPolymorphicWorkingDirectory(int sampleNumber)
    {
        var directories = new[]
        {
            "C:\\Windows\\System32",
            "C:\\Windows",
            "C:\\Windows\\Temp",
            "C:\\Temp",
            "C:\\Users\\Public",
            "C:\\ProgramData",
            "C:\\Windows\\System32\\config",
            "C:\\Windows\\Tasks",
            "%TEMP%",
            "%USERPROFILE%"
        };

        return directories[sampleNumber % directories.Length];
    }
}
